/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.runtime;

import com.sap.cds.feature.mt.lib.subscription.DataSourceInfo;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HanaConnectionProvider implements ConnectionProvider {
  private static Logger logger = LoggerFactory.getLogger(HanaConnectionProvider.class);

  @Override
  public Connection getConnection(
      String tenantId, DataSourceAndInfo dataSourceAndInfo, DataSourceInfo libContainerInfo)
      throws SQLException {
    DataSourceInfo info = dataSourceAndInfo.getDataSourceInfo();
    Connection connection = dataSourceAndInfo.getDataSource().getConnection(); // NOSONAR
    connection.setClientInfo("LOCALE", null); // workaround for HANA bug
    if (info.getUser().equals(connection.getMetaData().getUserName())) {
      logger.debug("Connection belongs to requested user, no reconnect needed");
      return connection;
    }
    try (Statement statement = connection.createStatement()) {
      logger.debug("Reconnect connection for user {}", info.getUser());
      statement.execute(
          "connect %s password \"%s\" ".formatted(info.getUser(), info.getPassword()));
      var userName = connection.getMetaData().getUserName();
      if (!info.getUser().equals(userName)) {
        throw new SQLException(
            "User should be %s after reconnect but is %s".formatted(info.getUser(), userName));
      }
      logger.debug("Set schema to {}", info.getSchema());
      connection.setSchema(info.getSchema());
      assert connection.getSchema().equals(info.getSchema());
    } catch (SQLException sqlException) {
      connection.close();
      throw sqlException;
    }
    return connection;
  }
}
