/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import org.apache.commons.lang3.StringUtils;

public abstract class DbCredentials {
  protected static final String JDBC = "jdbc:";
  private final String user;
  private final String password;
  private final String host;
  private final String port;
  private final String driver;
  protected final URI uri;

  protected DbCredentials(
      String user, String password, String host, String port, String driver, String uriStr)
      throws InternalError {
    this.user = user;
    this.password = password;
    this.host = host;
    this.port = port;
    this.driver = driver;
    if (StringUtils.isNotEmpty(uriStr)) {
      uri = getUri(uriStr);
    } else {
      uri = null;
    }
  }

  protected DbCredentials(DbCredentials dbCredentials) {
    this.user = dbCredentials.user;
    this.password = dbCredentials.password;
    this.host = dbCredentials.host;
    this.port = dbCredentials.port;
    this.driver = dbCredentials.driver;
    this.uri = dbCredentials.uri;
  }

  public abstract String getDatabaseId();

  public final String getUrl() {
    return buildUrl();
  }

  public abstract DbIdentifiers.DB getDB();

  public String getUser() {
    if (StringUtils.isNotEmpty(user)) {
      return user;
    } else {
      if (uri == null) {
        return "";
      }
      UserAndPassword userAndPassword = getUserFromUri(uri);
      return userAndPassword.user;
    }
  }

  public String getPassword() {
    if (StringUtils.isNotEmpty(password)) {
      return password;
    } else {
      if (uri == null) {
        return "";
      }
      UserAndPassword userAndPassword = getUserFromUri(uri);
      return userAndPassword.password;
    }
  }

  public String getDriver() {
    return driver;
  }

  public String getHost() {
    if (StringUtils.isNotEmpty(host)) {
      return host;
    } else {
      if (uri == null) {
        return "";
      }
      List<HostAndPort> hostAndPorts = getHostsFromUri(uri);
      if (!hostAndPorts.isEmpty()) {
        return hostAndPorts.get(0).host;
      } else {
        return "";
      }
    }
  }

  public String getPort() {
    if (StringUtils.isNotEmpty(port)) {
      return port;
    } else {
      if (uri == null) {
        return "";
      }
      List<HostAndPort> hostAndPorts = getHostsFromUri(uri);
      if (!hostAndPorts.isEmpty()) {
        return hostAndPorts.get(0).port;
      } else {
        return "";
      }
    }
  }

  public abstract DbCredentials createCopy();

  private URI getUri(String uriStr) throws InternalError {
    if (StringUtils.isEmpty(uriStr)) {
      throw new InternalError("No uri set");
    }
    try {
      if (uriStr.startsWith(JDBC)) {
        return new URI(uriStr.substring(5));
      } else {
        return new URI(uriStr);
      }
    } catch (URISyntaxException e) {
      throw new InternalError(e);
    }
  }

  protected abstract String buildUrl();

  protected abstract List<HostAndPort> getHostsFromUri(URI uri);

  protected abstract UserAndPassword getUserFromUri(URI uri);

  protected static class HostAndPort {
    String host;
    String port;
  }

  protected static class UserAndPassword {
    String user;
    String password;
  }
}
