/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.ParameterError;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.StringUtils;

public abstract class SqlOperations {
  private static final String WRONG_USER_OR_PASSWORD = "28000"; // NOSONAR
  private static final String INVALID_SCHEMA_NAME = "3F000";
  private static final String SCHEMA_NOT_FOUND = "90079";

  protected String dummySelectStatement;

  public static SqlOperations build(DbIdentifiers.DB db) throws InternalError {
    switch (db) {
      case H2:
        return new SqlOperationsH2();
      case POSTGRESQL:
        return new SqlOperationsPostgreSQL();
      case HANA:
        return new SqlOperationsHana();
      case SQLITE:
        return new SqlOperationsSqLite();
      default:
        throw new InternalError("Not supported DB" + db.toString());
    }
  }

  public void createSchema(String tenantId, Connection connection) throws SQLException {
    try (Statement createSchema = connection.createStatement()) {
      createSchema.execute(getCreateSchemaStatement(tenantId));
    } catch (ParameterError parameterError) {
      throw new SQLException(parameterError);
    }
  }

  public void deleteSchema(String tenantId, Connection connection) throws SQLException {
    try (Statement deleteSchema = connection.createStatement()) {
      deleteSchema.execute(getDeleteSchemaStatement(tenantId));
    } catch (ParameterError parameterError) {
      throw new SQLException(parameterError);
    }
  }

  public boolean doesSchemaExist(String tenantId, Connection connection) throws SQLException {
    return getAllSchemas(connection).stream().anyMatch(s -> s.equals(tenantId));
  }

  public List<String> getAllSchemas(Connection connection) throws SQLException {
    List<String> schemas = new ArrayList<>();
    try (Statement selectSchemas = connection.createStatement()) {
      selectSchemas.execute(getSelectSchemaStatement());
      try (ResultSet rs = selectSchemas.getResultSet()) {
        while (rs.next()) {
          schemas.add(rs.getString(1));
        }
      }
    }
    return schemas;
  }

  public void dummySelect(Connection connection) throws SQLException {
    try (Statement dummySelect = connection.createStatement()) {
      if (dummySelectStatement == null || dummySelectStatement.isEmpty()) {
        dummySelect.execute(getDummySelect());
      } else {
        dummySelect.execute(dummySelectStatement);
      }
    }
  }

  public void setDummySelectStatement(String dummySelectStatement) {
    this.dummySelectStatement = dummySelectStatement;
  }

  public boolean isAuthenticationProblem(String sqlState) {
    if (StringUtils.isBlank(sqlState)) {
      return false;
    }
    return sqlState.equals(WRONG_USER_OR_PASSWORD)
        || sqlState.equals(INVALID_SCHEMA_NAME)
        || sqlState.equals(SCHEMA_NOT_FOUND);
  }

  protected String getCreateSchemaStatement(String tenantId) throws ParameterError {
    Tools.checkTenantIdCharacters(tenantId);
    return "CREATE SCHEMA \"%s\"".formatted(tenantId);
  }

  protected String getDeleteSchemaStatement(String tenantId) throws ParameterError {
    Tools.checkTenantIdCharacters(tenantId);
    return "DROP SCHEMA IF EXISTS \"%s\" CASCADE".formatted(tenantId);
  }

  protected String getSelectSchemaStatement() {
    return "SELECT schema_name FROM information_schema.schemata";
  }

  protected String getDummySelect() {
    return "select 1 ";
  }
}
