/*
 * © 2023-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.adapter.subscription;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DatabindException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.mt.SaasClient;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.mt.DeploymentService;
import com.sap.cds.services.mt.SaasRegistryCallback;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ExecutorUtils;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Predicate;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Servlet providing the Saas Registry / CIS subscription endpoints. */
public class SaasProvisioningServlet extends HttpServlet {

  public static final String HEADER_STATUS_CALLBACK = "STATUS_CALLBACK";
  private static final long serialVersionUID = 1L;
  private static final ObjectMapper mapper = new ObjectMapper();
  private static final Logger logger = LoggerFactory.getLogger(SaasProvisioningServlet.class);
  private static final String DEPENDENCIES = "/dependencies";
  private static final String TENANTS = "/tenants/";
  private static final String SUCCEEDED = "SUCCEEDED";
  private static final String FAILED = "FAILED";

  private final CdsRuntime runtime;
  private final DeploymentService deploymentService;
  private final SaasClient saas;

  private final boolean isInternalUserAccessEnabled;

  public SaasProvisioningServlet(CdsRuntime runtime) {
    this.runtime = runtime;
    this.isInternalUserAccessEnabled =
        runtime
            .getEnvironment()
            .getCdsProperties()
            .getSecurity()
            .getAuthentication()
            .getInternalUserAccess()
            .isEnabled();
    this.deploymentService =
        runtime
            .getServiceCatalog()
            .getService(DeploymentService.class, DeploymentService.DEFAULT_NAME);
    ServiceBinding saasBinding = SaasClient.findBinding(runtime).orElse(null);
    this.saas = saasBinding != null ? new SaasClient(saasBinding) : null;
  }

  private static String getTenantId(HttpServletRequest req) {
    String tenantId = req.getPathInfo().split("/")[2];
    if (StringUtils.isEmpty(tenantId)) {
      throw new ErrorStatusException(ErrorStatuses.NOT_FOUND);
    }
    return tenantId;
  }

  private static void setContentType(HttpServletResponse resp, ContentType contType) {
    resp.setContentType(contType.getMimeType());
    resp.setCharacterEncoding(contType.getCharset().toString());
  }

  private static Map<String, Object> toMap(InputStream stream) throws IOException {
    try {
      TypeReference<Map<String, Object>> typeRef = new TypeReference<Map<String, Object>>() {};
      return mapper.readValue(stream, typeRef);
    } catch (DatabindException e) {
      throw new ErrorStatusException(ErrorStatuses.BAD_REQUEST, e);
    }
  }

  private static void handleSaasRegistryRaceCondition(long startTime) {
    long raceConditionWaitTime = 10000; // 10s
    long elapsed = System.currentTimeMillis() - startTime;
    // process might have happened too fast, saas-registry might not yet be ready to
    // accept status
    // reports
    if (elapsed < raceConditionWaitTime) {
      try {
        Thread.sleep(Math.max(raceConditionWaitTime - elapsed, 0));
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
    }
  }

  private static void handleException(HttpServletResponse res, Locale locale, ServiceException e) {
    if (e.getErrorStatus().getHttpStatus() >= 500 && e.getErrorStatus().getHttpStatus() < 600) {
      logger.error("Unexpected error", e);
    } else {
      logger.debug("Service exception thrown", e);
    }
    res.setStatus(e.getErrorStatus().getHttpStatus());
    try {
      String message = e.getLocalizedMessage(locale);
      if (message != null) {
        try (PrintWriter writer = res.getWriter()) {
          writer.write(message);
        }
      }
    } catch (IOException e1) {
      logger.error("Failed to write error message to response", e1);
    }
  }

  @Override
  protected void doGet(HttpServletRequest req, HttpServletResponse res) {
    processRequest(
        req,
        res,
        p -> DEPENDENCIES.equals(p),
        () -> {
          List<Map<String, Object>> dependencies = deploymentService.dependencies();
          setContentType(res, ContentType.APPLICATION_JSON);
          res.setStatus(HttpServletResponse.SC_OK);
          res.getWriter().write(mapper.writeValueAsString(dependencies));
        });
  }

  @Override
  protected void doPut(HttpServletRequest req, HttpServletResponse res) {
    processRequest(
        req,
        res,
        p -> p.startsWith(TENANTS),
        () -> {
          String tenantId = getTenantId(req);

          logger.info("Creating subscription for tenant '{}'", tenantId);
          Map<String, Object> payload = toMap(req.getInputStream());

          String callbackUrl = req.getHeader(HEADER_STATUS_CALLBACK);

          if (callbackUrl != null) {
            assertSaasClient();

            logger.debug("Processing subscription for tenant '{}' asynchronously", tenantId);
            long startTime = System.currentTimeMillis();
            ExecutorUtils.runAsynchronously(
                runtime,
                () -> {
                  boolean success = false;
                  try {
                    deploymentService.subscribe(tenantId, payload);
                    success = true;
                    logger.info("Subscription for tenant '{}' finished successfully", tenantId);
                  } catch (Throwable e) {
                    logger.error("Subscription for tenant '{}' failed", tenantId, e);
                  }

                  try {
                    handleSaasRegistryRaceCondition(startTime);

                    saas.callBackSaasRegistry(
                        createPayload(
                            success,
                            "Subscription",
                            deploymentService.appUiUrl(
                                tenantId, (String) payload.get("subscribedSubdomain"), payload)),
                        callbackUrl);
                  } catch (Throwable e) {
                    logger.error(
                        "Failed to report status for tenant '{}' to SaaS Registry", tenantId, e);
                  }
                });
            res.setStatus(HttpServletResponse.SC_ACCEPTED);
          } else {
            logger.debug("Processing subscription for tenant '{}' synchronously", tenantId);
            deploymentService.subscribe(tenantId, payload);
            res.setStatus(HttpServletResponse.SC_CREATED);
            setContentType(res, ContentType.TEXT_PLAIN);
            res.getWriter()
                .write(
                    deploymentService.appUiUrl(
                        tenantId, (String) payload.get("subscribedSubdomain"), payload));
          }
        });
  }

  @Override
  protected void doDelete(HttpServletRequest req, HttpServletResponse res) {
    processRequest(
        req,
        res,
        p -> p.startsWith(TENANTS),
        () -> {
          String tenantId = getTenantId(req);

          logger.info("Deleting subscription for tenant '{}'", tenantId);
          Map<String, Object> payload = toMap(req.getInputStream());

          String callbackUrl = req.getHeader(HEADER_STATUS_CALLBACK);
          if (callbackUrl != null) {
            assertSaasClient();

            logger.debug("Processing unsubscription for tenant '{}' asynchronously", tenantId);
            long startTime = System.currentTimeMillis();
            ExecutorUtils.runAsynchronously(
                runtime,
                () -> {
                  boolean success = false;
                  try {
                    deploymentService.unsubscribe(tenantId, payload);
                    success = true;
                    logger.info("Unsubscription for tenant '{}' finished successfully", tenantId);
                  } catch (Throwable e) {
                    logger.error("Unsubscription for tenant '{}' failed", tenantId, e);
                  }

                  try {
                    handleSaasRegistryRaceCondition(startTime);

                    saas.callBackSaasRegistry(
                        createPayload(success, "Unsubscription ", null), callbackUrl);
                  } catch (Throwable e) {
                    logger.error(
                        "Failed to report status for tenant '{}' to SaaS Registry", tenantId, e);
                  }
                });
            res.setStatus(HttpServletResponse.SC_ACCEPTED);
          } else {
            logger.debug("Processing unsubscription for tenant '{}' synchronously", tenantId);
            deploymentService.unsubscribe(tenantId, payload);
            res.setStatus(HttpServletResponse.SC_NO_CONTENT);
          }
        });
  }

  private void assertSaasClient() {
    if (saas == null) {
      logger.error("Asynchronous callbacks to SaaS Registry require a saas-registry binding.");
      throw new ErrorStatusException(ErrorStatuses.SERVER_ERROR);
    }
  }

  private void checkAuthorization() {
    UserInfo userInfo = runtime.getProvidedUserInfo();

    if (userInfo.isPrivileged() || (isInternalUserAccessEnabled && userInfo.isInternalUser())) {
      return;
    }

    String callbackScope =
        runtime
            .getEnvironment()
            .getCdsProperties()
            .getMultiTenancy()
            .getSecurity()
            .getSubscriptionScope();
    if (!userInfo.hasRole(callbackScope)) {
      throw new ErrorStatusException(ErrorStatuses.FORBIDDEN);
    }
  }

  private void processRequest(
      HttpServletRequest req,
      HttpServletResponse res,
      Predicate<String> pathMatcher,
      Processor processor) {
    if (pathMatcher.test(req.getPathInfo())) {
      try {
        checkAuthorization();
      } catch (ServiceException e) {
        handleException(res, null, e);
        return;
      } catch (Throwable t) {
        logger.error("Unexpected error", t);
        res.setStatus(HttpStatus.SC_INTERNAL_SERVER_ERROR);
        return;
      }

      runtime
          .requestContext()
          .systemUserProvider()
          .featureToggles(FeatureTogglesInfo.all())
          .run(
              requestContext -> {
                try {
                  processor.process();
                } catch (ServiceException e) {
                  handleException(res, requestContext.getParameterInfo().getLocale(), e);
                } catch (Throwable t) {
                  logger.error("Unexpected error", t);
                  res.setStatus(HttpStatus.SC_INTERNAL_SERVER_ERROR);
                }
              });
    } else {
      res.setStatus(HttpStatus.SC_NOT_FOUND);
    }
  }

  private SaasRegistryCallback createPayload(
      boolean success, String msgPrefix, String subscriptionUrl) {
    SaasRegistryCallback callback = SaasRegistryCallback.create();
    if (success) {
      callback.setStatus(SUCCEEDED);
      callback.setMessage(msgPrefix + " succeeded");
    } else {
      callback.setStatus(FAILED);
      callback.setMessage(msgPrefix + " failed");
    }
    callback.setSubscriptionUrl(subscriptionUrl);

    return callback;
  }

  @FunctionalInterface
  private static interface Processor {

    void process() throws IOException;
  }
}
