/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.runtime;

import com.sap.cds.feature.mt.lib.subscription.DataSourceInfo;
import com.sap.cds.feature.mt.lib.subscription.DbIdentifiers;
import com.sap.cds.feature.mt.lib.subscription.InstanceLifecycleManager;
import com.sap.cds.feature.mt.lib.subscription.MtxTools;
import com.sap.cds.feature.mt.lib.subscription.SqlOperations;
import com.sap.cds.feature.mt.lib.subscription.TenantMutexFactory;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.UnknownTenant;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLInvalidAuthorizationSpecException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import javax.sql.DataSource;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class DataSourceLookup {

  private static final Logger logger = LoggerFactory.getLogger(DataSourceLookup.class);
  private final ConcurrentHashMap<String, DataSourceAndInfo> tenantToDataSource =
      new ConcurrentHashMap<>();
  private final InstanceLifecycleManager instanceLifecycleManager;
  private final LibContainerCache libContainerCache = new LibContainerCache();
  private boolean combinePools = false;

  protected DataSourceLookup(
      InstanceLifecycleManager instanceLifecycleManager, boolean combinePools) {
    this.instanceLifecycleManager = instanceLifecycleManager;
    this.combinePools = combinePools;
  }

  protected DataSourceLookup(InstanceLifecycleManager instanceLifecycleManager) {
    this(instanceLifecycleManager, false);
  }

  DataSourceAndInfo getDataSourceAndInfo(String tenantId) throws InternalError, UnknownTenant {
    logger.debug("Determine data source information for tenant {}", tenantId);
    DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
    if (dataSourceAndInfo != null) {
      logger.debug(
          "Found data source information for tenant {}: schema={}",
          tenantId,
          dataSourceAndInfo.getDataSourceInfo().getSchema());
      return dataSourceAndInfo;
    } else {
      synchronized (TenantMutexFactory.get(tenantId)) {
        dataSourceAndInfo = tenantToDataSource.get(tenantId);
        if (dataSourceAndInfo != null) {
          logger.debug(
              "Found data source information for tenant {}: schema={}",
              tenantId,
              dataSourceAndInfo.getDataSourceInfo().getSchema());
          return dataSourceAndInfo;
        }
        logger.debug("Access instance manager for tenant {}", tenantId);
        DataSourceInfo tenantDataSourceInfo =
            instanceLifecycleManager.getDataSourceInfo(tenantId, false);
        String dbKey = tenantDataSourceInfo.getDbKey();
        DataSource dataSource = null;
        if (combinePools) {
          logger.debug("Combine pools is active");
          dataSource = libContainerCache.get(dbKey);
          if (dataSource == null) {
            logger.debug("Datasource is null, prepare pool");
            preparePool(tenantDataSourceInfo);
            dataSource = libContainerCache.get(dbKey);
            if (dataSource == null) {
              throw new InternalError("Could not find database pool for db key " + dbKey);
            }
          }
        } else {
          // make sure that also in mode "one pool per tenant" lib containers are created
          preparePool(tenantDataSourceInfo);
          dataSource = create(tenantDataSourceInfo);
        }
        DataSourceAndInfo newEntry = new DataSourceAndInfo(dataSource, tenantDataSourceInfo);
        tenantToDataSource.put(tenantId, newEntry);
        return newEntry;
      }
    }
  }

  public boolean isAuthenticationProblem(SQLException sqlException) {
    if (sqlException instanceof SQLInvalidAuthorizationSpecException) {
      return true;
    }
    try {
      SqlOperations sqlOperations = SqlOperations.build(instanceLifecycleManager.getDbType());
      return sqlOperations.isAuthenticationProblem(determineSqlState(sqlException));
    } catch (InternalError e) {
      logger.error("Not supported DB", e);
      // Not supported DB, should never happen
      return false;
    }
  }

  DataSourceInfo getLibContainerInfo(String dbkey) {
    return libContainerCache.getInfo(dbkey);
  }

  protected String determineSqlState(Throwable exception) {
    if (exception instanceof SQLException sqlException) {
      String sqlState = sqlException.getSQLState();
      if (StringUtils.isNotBlank(sqlState)) {
        return sqlState;
      }
    }
    if (exception.getCause() != null) {
      return determineSqlState(exception.getCause());
    }
    return "";
  }

  private synchronized void preparePool(DataSourceInfo dataSourceInfo) throws InternalError {
    logger.debug("Prepare pool for database key {}", dataSourceInfo.getDbKey());
    if (libContainerCache.isContained(dataSourceInfo.getDbKey())) {
      return;
    }
    List<DataSourceInfo> libContainers =
        instanceLifecycleManager.createAndGetLibContainers(dataSourceInfo);
    libContainers.stream().forEach(this::createAndCachePoolsForLibContainerPools);
  }

  private void createAndCachePoolsForLibContainerPools(DataSourceInfo libContainerInfo) {
    libContainerCache.createIfNotExist(
        libContainerInfo.getDbKey(), () -> create(libContainerInfo), libContainerInfo);
  }

  void fixDataSourceAfterCredentialChange(String tenantId, DataSource usedPool)
      throws InternalError {
    if (StringUtils.isBlank(tenantId)) {
      logger.debug("No tenant specified");
      throw new InternalError("No tenant specified");
    }
    synchronized (TenantMutexFactory.get(tenantId)) {
      // Refresh only, if the container was really recreated for the same tenant
      // A parallel task could have fixed it already.
      // It must be avoided that pools that were fixed and are used are accidentally
      // deleted
      DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
      if (dataSourceAndInfo == null || usedPool != dataSourceAndInfo.getDataSource()) {
        // already removed from cache, or new pool created
        return;
      }
      DataSourceInfo cachedInfo = dataSourceAndInfo.getDataSourceInfo();
      DataSourceInfo freshInfo;
      try {
        freshInfo = instanceLifecycleManager.getDataSourceInfo(tenantId, true);
      } catch (UnknownTenant unknownTenant) {
        freshInfo = null;
      }
      if (freshInfo == null) {
        adjustCacheAfterTenantDeletion(tenantId);
        logger.debug("Tenant was deleted");
        throw new InternalError("Tenant was deleted");
      }
      if (credentialsChanged(cachedInfo, freshInfo)) {
        adjustCacheAfterCredentialChange(tenantId, dataSourceAndInfo, freshInfo);
      } else {
        // credentials haven't changed => nothing to fix
        logger.debug("Normal database error");
        throw new InternalError("Normal database error");
      }
    }
  }

  private void adjustCacheAfterCredentialChange(
      String tenantId, DataSourceAndInfo dataSourceAndInfo, DataSourceInfo currentInfo) {
    if (!combinePools
        || !Objects.equals(
            dataSourceAndInfo.getDataSourceInfo().getDatabaseId(), currentInfo.getDatabaseId())
        || !Objects.equals(
            dataSourceAndInfo.getDataSourceInfo().getDbKey(), currentInfo.getDbKey())) {
      deleteFromCache(tenantId);
    } else {
      // credentials must be updated
      dataSourceAndInfo.setDataSourceInfo(currentInfo);
    }
  }

  private boolean credentialsChanged(DataSourceInfo cached, DataSourceInfo current) {
    return !Objects.equals(current.getPassword(), cached.getPassword())
        || !Objects.equals(current.getUser(), cached.getUser())
        || !Objects.equals(current.getSchema(), cached.getSchema())
        || !Objects.equals(current.getDbKey(), cached.getDbKey());
  }

  private void adjustCacheAfterTenantDeletion(String tenantId) {
    if (!combinePools) {
      deleteFromCache(tenantId);
    } else {
      tenantToDataSource.remove(tenantId);
    }
  }

  void deleteFromCache(String tenantId) {
    if (tenantId == null) {
      return;
    }
    logger.debug("Instance for tenant {} deleted from cache", tenantId);
    synchronized (TenantMutexFactory.get(tenantId)) {
      DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
      if (dataSourceAndInfo == null) {
        return;
      }
      tenantToDataSource.remove(tenantId);
      if (combinePools) {
        long numberOfEntries =
            tenantToDataSource.entrySet().stream()
                .filter(e -> e.getValue().getDataSource() == dataSourceAndInfo.getDataSource())
                .count();
        if (numberOfEntries == 0) {
          libContainerCache.remove(dataSourceAndInfo.getDataSourceInfo().getDbKey());
          closeDataSource(dataSourceAndInfo.getDataSource());
        }
      } else {
        closeDataSource(dataSourceAndInfo.getDataSource());
      }
    }
  }

  public void reset() {
    if (combinePools) {
      tenantToDataSource.clear();
      libContainerCache.reset();
    } else {
      tenantToDataSource.entrySet().stream()
          .forEach(e -> closeDataSource(e.getValue().getDataSource()));
      tenantToDataSource.clear();
    }
  }

  public List<HealthCheckResult> checkDataSourcePerDb(String dummySelectStatement) {
    List<HealthCheckResult> result = new ArrayList<>();
    try {
      instanceLifecycleManager.getLibContainers().stream()
          .forEach(this::createAndCachePoolsForLibContainerPools);
    } catch (InternalError e) {
      logger.error("Could not access containers owned by the mt-lib", e);
      HealthCheckResult healthCheckResult = new HealthCheckResult("", false, e);
      result.add(healthCheckResult);
      return result;
    }
    libContainerCache.stream()
        .forEach(
            e -> {
              try (Connection connection = e.getValue().getConnection()) {
                SqlOperations sqlOperations =
                    SqlOperations.build(instanceLifecycleManager.getDbType());
                sqlOperations.setDummySelectStatement(dummySelectStatement);
                sqlOperations.dummySelect(connection);
                HealthCheckResult healthCheckResult = new HealthCheckResult(e.getKey(), true, null);
                result.add(healthCheckResult);
              } catch (InternalError | SQLException exception) {
                HealthCheckResult healthCheckResult =
                    new HealthCheckResult(e.getKey(), false, exception);
                result.add(healthCheckResult);
              }
            });
    return result;
  }

  protected abstract DataSource create(DataSourceInfo info) throws InternalError;

  protected abstract void closeDataSource(DataSource dataSource);

  public boolean isCombinePools() {
    return combinePools;
  }

  public PoolMode poolMode() {
    if (combinePools) {
      return PoolMode.POOL_PER_DB_CREDENTIALS_PER_TENANT;
    } else {
      return PoolMode.POOL_AND_CREDENTIALS_PER_TENANT;
    }
  }

  public DbIdentifiers.DB getDbType() {
    return instanceLifecycleManager.getDbType();
  }

  public boolean healthCheckPossible() {
    try {
      return !instanceLifecycleManager.getLibContainers().isEmpty();
    } catch (InternalError e) {
      return false;
    }
  }

  private class LibContainerCache {

    private final ConcurrentHashMap<String, DataSource> dbKeyToLibContainer =
        new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, DataSourceInfo> dbKeyToLibContainerInfo =
        new ConcurrentHashMap<>();

    private boolean isContained(String dbKey) {
      return dbKeyToLibContainer.containsKey(dbKey);
    }

    private DataSource get(String dbKey) {
      return dbKeyToLibContainer.get(dbKey);
    }

    private DataSourceInfo getInfo(String dbKey) {
      return dbKeyToLibContainerInfo.get(dbKey);
    }

    private DataSource remove(String dbKey) {
      dbKeyToLibContainerInfo.remove(dbKey);
      return dbKeyToLibContainer.remove(dbKey);
    }

    private void reset() {
      dbKeyToLibContainer.entrySet().stream().forEach(e -> closeDataSource(e.getValue()));
      dbKeyToLibContainer.clear();
      dbKeyToLibContainerInfo.clear();
    }

    private Stream<Map.Entry<String, DataSource>> stream() {
      return dbKeyToLibContainer.entrySet().stream();
    }

    private void createIfNotExist(
        String dbKey,
        MtxTools.SupplierWithInternalError<DataSource> dataSourceSupplier,
        DataSourceInfo dataSourceInfo) {
      try {
        dbKeyToLibContainerInfo.putIfAbsent(dbKey, dataSourceInfo);
        dbKeyToLibContainer.putIfAbsent(dbKey, dataSourceSupplier.get());
      } catch (InternalError e) {
        logger.error("Cannot create data source pool", e);
      }
    }
  }
}
