/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.hana.mt.service.HanaMtService;
import com.sap.cds.services.utils.lib.tools.api.ResilienceConfig;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class InstanceLifecycleManagerBuilder {

  private ServiceManager serviceManager;
  private HanaMtService hanaMtService;
  private DbIdentifiers dbIdentifiers;
  private List<DbCredentials> dbCredentialsList = new ArrayList<>();
  private DbIdentifiers.DB db = DbIdentifiers.DB.NONE;
  private Duration smCacheRefreshInterval;
  private ResilienceConfig serviceManagerCacheResilienceConfig = ResilienceConfig.NONE;
  // maps the service manager destination to the instance lifecycle manager instance. This assures
  // that only one ILM is used per
  // bound service manager service.
  private static ConcurrentHashMap<String, InstanceLifecycleManagerImpl> smIdToILM =
      new ConcurrentHashMap<>();
  private boolean acceptInstancesWithoutTenant;
  private boolean ignoreDuplicateTenantInstances;

  public InstanceLifecycleManager build() throws InternalError {
    if (serviceManager != null || hanaMtService != null) {
      if (dbIdentifiers != null && !(dbIdentifiers instanceof DbIdentifiersHana)) {
        throw new InternalError("Hana database specified with database identifiers of non-HANA DB");
      }
      // Use Hana DB as default
      if (dbIdentifiers == null) {
        dbIdentifiers = new DbIdentifiersHana(new HashSet<>());
      }
      HanaAccess hanaAccess;
      String serviceBindingName;
      if (hanaMtService != null) {
        hanaAccess = hanaMtService;
        serviceBindingName = hanaMtService.getServiceBindingName();
      } else {
        hanaAccess =
            new ServiceManagerCache(
                serviceManager,
                smCacheRefreshInterval,
                serviceManagerCacheResilienceConfig != null
                    ? serviceManagerCacheResilienceConfig
                    : ResilienceConfig.NONE,
                acceptInstancesWithoutTenant,
                ignoreDuplicateTenantInstances);
        serviceBindingName = serviceManager.getServiceBindingName();
      }

      return smIdToILM.computeIfAbsent(
          serviceBindingName,
          id -> new InstanceLifecycleManagerImpl(hanaAccess, (DbIdentifiersHana) dbIdentifiers));
    } else if (dbIdentifiers != null) {
      if (dbIdentifiers instanceof DbIdentifiersSql dbIdentifiersSql) {
        return new InstanceLifecycleManagerSqlDb(dbIdentifiersSql);
      } else if (dbIdentifiers instanceof DbIdentifiersSqLite dbIdentifiersSqLite) {
        return new InstanceLifecycleManagerSqLite((dbIdentifiersSqLite).getRoot());
      } else {
        throw new InternalError("No database credentials provided");
      }
    } else if (!dbCredentialsList.isEmpty()) {
      if (db == DbIdentifiers.DB.NONE) {
        throw new InternalError("No database type set");
      }
      return new InstanceLifecycleManagerSqlDb(new DbIdentifiersSql(dbCredentialsList));
    } else {
      throw new InternalError("No instance manager, service manager or databases specified");
    }
  }

  public static InstanceLifecycleManagerBuilder create() {
    return new InstanceLifecycleManagerBuilder();
  }

  private InstanceLifecycleManagerBuilder() {}

  public InstanceLifecycleManagerBuilder serviceManager(ServiceManager serviceManager) {
    this.serviceManager = serviceManager;
    return this;
  }

  public InstanceLifecycleManagerBuilder hanaMtService(HanaMtService hanaMtService) {
    this.hanaMtService = hanaMtService;
    return this;
  }

  public InstanceLifecycleManagerBuilder dbIdentifiers(DbIdentifiers dbIdentifiers) {
    this.dbIdentifiers = dbIdentifiers;
    return this;
  }

  public InstanceLifecycleManagerBuilder addDbCredentials(Map<String, Object> credentials)
      throws InternalError {
    return addDbCredentials(DbIdentifiers.DB.NONE, credentials);
  }

  public InstanceLifecycleManagerBuilder smCacheRefreshInterval(Duration smCacheRefreshInterval) {
    this.smCacheRefreshInterval = smCacheRefreshInterval;
    return this;
  }

  public InstanceLifecycleManagerBuilder addDbCredentials(
      DbIdentifiers.DB db, Map<String, Object> credentials) throws InternalError {
    DbCredentialsBuilder dbCredentialsBuilder =
        DbCredentialsBuilder.create().db(db).credentials(credentials);
    DbCredentials dbCredentials = dbCredentialsBuilder.build();
    dbCredentialsList.add(dbCredentials);
    if (this.db != DbIdentifiers.DB.NONE && this.db != dbCredentials.getDB()) {
      throw new InternalError(
          "It is not possible to combine different types of databases for multi tenancy");
    }
    if (this.db == DbIdentifiers.DB.NONE) {
      this.db = dbCredentials.getDB();
    }
    return this;
  }

  public InstanceLifecycleManagerBuilder smCacheResilienceConfig(
      ResilienceConfig resilienceConfig) {
    if (resilienceConfig != null) {
      this.serviceManagerCacheResilienceConfig = resilienceConfig;
    }
    return this;
  }

  public InstanceLifecycleManagerBuilder acceptInstancesWithoutTenant(
      boolean acceptInstancesWithoutTenant) {
    this.acceptInstancesWithoutTenant = acceptInstancesWithoutTenant;
    return this;
  }

  public InstanceLifecycleManagerBuilder ignoreDuplicateTenantInstances(
      boolean ignoreDuplicateTenantInstances) {
    this.ignoreDuplicateTenantInstances = ignoreDuplicateTenantInstances;
    return this;
  }
}
