/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import com.sap.cds.feature.mt.lib.subscription.ServiceOperation.Status;
import com.sap.cds.feature.mt.lib.subscription.ServiceOperation.Type;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.UnknownTenant;
import com.sap.cds.services.utils.lib.tools.api.UuidChecker;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InstanceLifecycleManagerImpl implements InstanceLifecycleManager {

  private static final Logger logger = LoggerFactory.getLogger(InstanceLifecycleManagerImpl.class);
  private static final String HOST = "host";
  public static final String CREATION_SUCCEEDED = "CREATION_SUCCEEDED";
  private final HanaAccess hanaAccess;
  private Set<String> dbIds = new HashSet<>();

  /**
   * @param hanaAccess the service manager/hana mt service
   * @param dbIdentifiers optional provided DB identifiers
   */
  InstanceLifecycleManagerImpl(HanaAccess hanaAccess, DbIdentifiersHana dbIdentifiers) {
    if (dbIdentifiers != null) {
      dbIds.addAll(dbIdentifiers.getDbIds());
    }
    this.hanaAccess = hanaAccess;
  }

  @Override
  public void createNewInstance(
      String tenantId,
      ProvisioningParameters instanceParameters,
      BindingParameters bindingParameters)
      throws InternalError {
    if (instanceParameters != null && !instanceParameters.isEmpty()) {
      String databaseId = (String) instanceParameters.get(DATABASE_ID);
      if (databaseId != null) {
        logger.debug("Using database id {}", databaseId);
      }
    }
    hanaAccess.createInstance(tenantId, instanceParameters, bindingParameters);
  }

  @Override
  public void deleteInstance(String tenantId) throws InternalError {
    try {
      checkThatTenantExists(tenantId);
    } catch (UnknownTenant unknownTenant) {
      logger.warn("No HDI container for tenant {} found", tenantId);
      return;
    }
    hanaAccess.deleteInstance(tenantId);
  }

  @Override
  public DataSourceInfo getDataSourceInfo(String tenantId, boolean forceCacheUpdate)
      throws InternalError, UnknownTenant {
    return getDataSourceInfoInternal(tenantId, forceCacheUpdate);
  }

  protected DataSourceInfo getDataSourceInfoInternal(String tenantId, boolean forceCacheUpdate)
      throws InternalError, UnknownTenant {
    ServiceInstance instance = null;
    try {
      instance =
          hanaAccess
              .getInstance(tenantId, forceCacheUpdate)
              .orElseThrow(() -> new UnknownTenant("Tenant [" + tenantId + "] is not known"));
    } catch (InternalError e) {
      throw new UnknownTenant(e, "Tenant [" + tenantId + "] is not known");
    }
    Map<String, Object> credentials;
    var binding =
        instance
            .getBinding()
            .orElseThrow(
                () ->
                    new InternalError(
                        "Database container service instance for tenant %s doesn't have a ready binding"
                            .formatted(tenantId)));
    credentials = binding.getCredentials();
    return DataSourceInfoBuilder.createBuilder()
        .host((String) credentials.get(HOST))
        .port((String) credentials.get("port"))
        .driver((String) credentials.get("driver"))
        .url((String) credentials.get("url"))
        .schema((String) credentials.get("schema"))
        .hdiUser((String) credentials.get("hdi_user"))
        .hdiUserName((String) credentials.get("hdi_user_name"))
        .hdiPassword((String) credentials.get("hdi_password"))
        .user((String) credentials.get("user"))
        .userName((String) credentials.get("user_name"))
        .password((String) credentials.get("password"))
        .credentialType((String) credentials.get("credential-type"))
        .certificate((String) credentials.get("certificate"))
        .tenantId(tenantId)
        .id(instance.getId())
        .statusAsText(CREATION_SUCCEEDED)
        .dbKey(createDbKey((String) credentials.get(HOST), (String) credentials.get("port")))
        .databaseId(getDatabaseId(credentials))
        .build();
  }

  @Override
  public ContainerStatus getContainerStatus(String tenantId) throws InternalError {
    return hanaAccess.getInstance(tenantId, true).stream()
        .map(
            instance -> {
              var status = instance.getLastOperation().getStatus();
              var type = instance.getLastOperation().getType();
              // As a container status "creation error" triggers a deletion, it is only set if in
              // addition the ready flag is not set.
              if (type == Type.CREATE && !instance.isReady() && status == Status.FAILED) {
                return ContainerStatus.CREATION_ERROR;
              }
              if (type == Type.CREATE && status == Status.IN_PROGRESS) {
                return ContainerStatus.CREATION_IN_PROGRESS;
              }
              // all other types do not determine the container state. Think for example of a patch
              // operation
              // that fails. The container is still working. Even if a delete operation fails the
              // container is still there.
              return ContainerStatus.OK;
            })
        .findFirst()
        .orElse(ContainerStatus.DOES_NOT_EXIST);
  }

  @Override
  public boolean hasCredentials(String tenantId, boolean forceCacheUpdate) throws InternalError {
    return hanaAccess.getInstance(tenantId, forceCacheUpdate).stream()
        .map(i -> i.getBinding().isPresent())
        .findFirst()
        .orElse(false);
  }

  @Override
  public Map<String, TenantMetadata> getAllTenantInfos(boolean forceCacheUpdate)
      throws InternalError {
    return hanaAccess.getAllTenants(forceCacheUpdate).stream()
        .filter(t -> t.isUsable() && FilterTenants.realTenants().test(t.tenantId()))
        .map(t -> new TenantMetadata(t.tenantId(), t.databaseId()))
        .collect(Collectors.toMap(TenantMetadata::getTenantId, Function.identity()));
  }

  @Override
  public void checkThatTenantExists(String tenantId) throws UnknownTenant, InternalError {
    if (hanaAccess.getInstance(tenantId, false).isEmpty()) {
      throw new UnknownTenant("Tenant " + tenantId + " is not known");
    }
  }

  @Override
  public List<DataSourceInfo> createAndGetLibContainers(DataSourceInfo dataSourceInfo)
      throws InternalError {
    if (dataSourceInfo == null) {
      throw new InternalError("Parameter dataSourceInfo is null");
    }
    String databaseId = dataSourceInfo.getDatabaseId();
    // determine missing container of DB used by dataSourceInfo
    if (isLibContainerMissing(databaseId)) {
      try {
        logger.debug("Create new mt-lib container for database {}", databaseId);
        createNewInstance(
            getMtLibContainerName(databaseId),
            createProvisioningParameters(databaseId),
            new BindingParameters());
      } catch (InternalError internalError) {
        logger.error(
            "Could not create new mt-lib container for database {} because of {} ",
            databaseId,
            internalError.getMessage());
      }
    }
    return getLibContainers();
  }

  @Override
  public List<DataSourceInfo> getLibContainers() {
    List<DataSourceInfo> dsInfo = new ArrayList<>();
    insertDbIdsFromHana();
    logger.debug("Determine mt-lib containers");
    dbIds.stream()
        .forEach(
            id -> {
              try {
                var mtLibContainerName = getMtLibContainerName(id);
                var dsInfoEntry = getDataSourceInfo(mtLibContainerName, false);
                logger.debug(
                    "Credentials for mt-lib container {} read. User = {}, hdi user = {}",
                    mtLibContainerName,
                    dsInfoEntry.getUser(),
                    dsInfoEntry.getHdiUser());
                dsInfo.add(dsInfoEntry);
              } catch (InternalError | UnknownTenant error) {
                // NOSONAR
              }
            });
    return dsInfo;
  }

  public void clearCache() {
    hanaAccess.clearCache();
  }

  private void insertDbIdsFromHana() {
    try {
      dbIds.addAll(hanaAccess.getDatabaseIds(false));
    } catch (InternalError e) {
      logger.error("Could not access SM", e);
    }
  }

  private boolean isLibContainerMissing(String dbId) {
    try {
      return hanaAccess.getInstance(getMtLibContainerName(dbId), false).isEmpty();
    } catch (InternalError e) {
      return true;
    }
  }

  @SuppressWarnings("unused")
  private DbIdentifiers getDbIdentifiers() {
    insertDbIdsFromHana();
    return new DbIdentifiersHana(dbIds); // NOSONAR
  }

  @Override
  public void insertDbIdentifiers(DbIdentifiers dbIdentifiers) {
    dbIds.addAll(((DbIdentifiersHana) dbIdentifiers).getDbIds());
  }

  @Override
  public DbIdentifiers.DB getDbType() {
    return DbIdentifiers.DB.HANA;
  }

  private String createDbKey(String host, String port) {
    return host + ":" + port;
  }

  private String getDatabaseId(Map<String, Object> credentials) {
    String databaseId = (String) credentials.get("database_id");
    if (StringUtils.isNotEmpty(databaseId)) {
      return databaseId;
    }
    databaseId = StringUtils.substringBefore((String) credentials.get(HOST), ".");
    return UuidChecker.isUUId(databaseId) ? databaseId : null;
  }
}
