/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription.hana.mt.service;

import static com.sap.cds.feature.mt.lib.subscription.hana.mt.service.Tools.asListMap;
import static com.sap.cds.feature.mt.lib.subscription.hana.mt.service.Tools.asMap;

import com.sap.cds.feature.mt.lib.subscription.ServiceBinding;
import com.sap.cds.feature.mt.lib.subscription.ServiceInstance;
import com.sap.cds.services.utils.lib.tools.api.QueryParameter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.lang3.StringUtils;

public class Container extends HashMap<String, Object> {

  private static final String LABELS_NODE = "labels";
  private static final String TENANT_ID = "tenant_id";
  private static final String STATUS_NODE = "status";
  private static final String ID_NODE = "id";
  private static final String CREDENTIALS_NODE = "credentials";
  private static final String TYPE_NODE = "type";
  private static final String HDI_TYPE = "hdi";
  private static final String MAKE_UNIQUE_NAME_NODE = "makeUniqueName";
  private static final String TENANT_ID_TEMPLATE = "tenant_id=%s";
  private static final String EXPAND_PARAMETER = "$expand";
  private static final String EXPAND_CREDENTIALS = "credentials";
  private static final String DATA_NODE = "data";
  private final List<Credential> credentials = new ArrayList<>();

  public Container(Map<String, ?> m) {
    super(m);
    if (containsKey(CREDENTIALS_NODE)) {
      asListMap(get(CREDENTIALS_NODE)).stream()
          .forEach(credentialData -> credentials.add(new Credential(credentialData)));
    }
  }

  public static Map<String, Object> createCreatePayload(String tenantId) {
    var payload = new HashMap<String, Object>();
    payload.put(TYPE_NODE, HDI_TYPE);
    payload.put(MAKE_UNIQUE_NAME_NODE, false);
    var labels = List.of(TENANT_ID_TEMPLATE.formatted(tenantId));
    payload.put(LABELS_NODE, labels);
    return payload;
  }

  public static QueryParameter getExpandCredentials() {
    return new QueryParameter(EXPAND_PARAMETER, EXPAND_CREDENTIALS);
  }

  public static List<Container> getContainersFromPayload(Map<String, Object> payload) {
    var containers = new ArrayList<Container>();
    if (payload.containsKey(DATA_NODE)) {
      asListMap(payload.get(DATA_NODE)).forEach(c -> containers.add(new Container(c)));
    }
    return containers;
  }

  public String getBtpTenant() throws InternalError {
    var labels = (List<String>) get(LABELS_NODE);
    if (labels == null || labels.isEmpty()) {
      throw new InternalError("No label section");
    }
    var labelOpt = labels.stream().filter(l -> l.contains(TENANT_ID)).findFirst();
    var label = labelOpt.orElseThrow(() -> new InternalError("No btp tenant id is set"));
    var parts = label.split("=");
    if (parts.length < 2 || StringUtils.isBlank(parts[1])) {
      throw new InternalError("No btp tenant id is set");
    }
    return parts[1];
  }

  public Status getStatus() {
    return new Status(asMap(get(STATUS_NODE)));
  }

  public String getId() {
    if (!containsKey(ID_NODE)) {
      return "";
    }
    return (String) get(ID_NODE);
  }

  public List<Credential> getCredentials() {
    return credentials;
  }

  public void insertCredentials(List<Credential> credentials) {
    this.credentials.addAll(credentials);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    if (!super.equals(o)) {
      return false;
    }
    Container container = (Container) o;
    return Objects.equals(credentials, container.credentials);
  }

  @Override
  public int hashCode() {
    return Objects.hash(super.hashCode(), credentials);
  }

  public ServiceInstance toServiceInstance() {
    var instanceMap = new HashMap<String, Object>();
    instanceMap.put(ServiceInstance.ID, getId());
    instanceMap.put(ServiceInstance.USABLE, getStatus().isReady());
    var instance = new ServiceInstance(instanceMap);
    instance.insertTenant(getBtpTenant());
    instance.setBindings(toServiceBindings());
    return instance;
  }

  public List<ServiceBinding> toServiceBindings() {
    var bindings = new ArrayList<ServiceBinding>();
    if (credentials.isEmpty()) {
      return bindings;
    }
    credentials.forEach(
        credential -> {
          var bindingMap = new HashMap<String, Object>();
          bindingMap.put(ServiceBinding.ID, credential.getId());
          bindingMap.put(ServiceBinding.SERVICE_INSTANCE_ID, getId());
          // We must wait until a timestamp is included into the credentials
          // bindingMap.put(ServiceBinding.CREATED_AT, credential.getCreateTimeStamp());
          bindingMap.put(
              ServiceBinding.LABELS, Collections.singletonMap(TENANT_ID, getBtpTenant()));
          var targetCredentials = new HashMap<String, Object>();
          targetCredentials.put(ServiceBinding.CERTIFICATE, credential.getCertificate());
          targetCredentials.put(ServiceBinding.DATABASE_ID, credential.getDatabaseId());
          targetCredentials.put(ServiceBinding.DRIVER, credential.getDriver());
          targetCredentials.put(ServiceBinding.HDI_PASSWORD, credential.getHdiPassword());
          targetCredentials.put(ServiceBinding.HDI_USER, credential.getHdiUser());
          targetCredentials.put(ServiceBinding.HOST, credential.getHost());
          targetCredentials.put(ServiceBinding.PASSWORD, credential.getPassword());
          targetCredentials.put(ServiceBinding.PORT, credential.getPort());
          targetCredentials.put(ServiceBinding.SCHEMA, credential.getSchema());
          targetCredentials.put(ServiceBinding.URL, credential.getUrl());
          targetCredentials.put(ServiceBinding.USER, credential.getUser());
          bindingMap.put(ServiceBinding.CREDENTIALS, targetCredentials);
          bindings.add(new ServiceBinding(bindingMap));
        });
    return bindings;
  }
}
