/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription.hana.mt.service;

import static com.sap.cds.feature.mt.lib.subscription.hana.mt.service.Tools.asListMap;
import static com.sap.cds.feature.mt.lib.subscription.hana.mt.service.Tools.asMap;

import com.sap.cds.feature.mt.lib.subscription.ProvisioningParameters;
import com.sap.cds.services.utils.lib.tools.api.QueryParameter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.lang3.StringUtils;

public class HanaTenant extends HashMap<String, Object> {

  private static final String LABELS_NODE = "labels";
  private static final String TENANT_ID = "tenant_id";
  private static final String STATUS_NODE = "status";
  private static final String ID_NODE = "id";
  private static final String CONTAINERS_NODE = "containers";
  private static final String DATA_ENCRYPTION_NODE = "dataEncryption";
  private static final String MODE_NODE = "mode";
  private static final String TENANT_ID_TEMPLATE = TENANT_ID + "=%s";
  private static final String PREFIX_TEMPLATE = "prefix=%s";
  private static final String NAME_TEMPLATE = "name=%s";
  private static final String CONTAINER_FILTER_PARAMETER = "containerFilter";
  private static final String BTP_TENANT_LABEL_QUERY =
      "hassubset(labels, [\"" + TENANT_ID + "=%s\"])";
  private static final String EXPAND_PARAMETER = "$expand";
  private static final String EXPAND_CONTAINERS_AND_CREDENTIALS = "containers/credentials";
  private static final String DATA_NODE = "data";
  public static final String SERVICE_INSTANCE_ID_NODE = "serviceInstanceID";
  private final List<Container> containers = new ArrayList<>();

  public HanaTenant(Map<String, ?> m) {
    super(m);
    if (containsKey(CONTAINERS_NODE)) {
      asListMap(get(CONTAINERS_NODE)).stream()
          .forEach(containerData -> containers.add(new Container(containerData)));
    }
  }

  public static Map<String, Object> createCreatePayload(
      String tenantId, ProvisioningParameters provisioningParameters, String prefix, String name) {
    var payload = new HashMap<String, Object>();
    payload.put(SERVICE_INSTANCE_ID_NODE, provisioningParameters.getDatabaseId());
    var dataEncryption = (Map<String, Object>) provisioningParameters.get(DATA_ENCRYPTION_NODE);
    if (dataEncryption != null) {
      var mode = (String) dataEncryption.get(MODE_NODE);
      if (mode != null) {
        var dataEncryptionPayload = new HashMap<>();
        dataEncryptionPayload.put(MODE_NODE, mode);
        payload.put(DATA_ENCRYPTION_NODE, dataEncryptionPayload);
      }
    }
    // maximum label size is 64
    var prefixLabel = PREFIX_TEMPLATE.formatted(prefix);
    prefixLabel = prefixLabel.length() > 64 ? prefixLabel.substring(0, 64) : prefixLabel;
    var nameLabel = NAME_TEMPLATE.formatted(name);
    nameLabel = nameLabel.length() > 64 ? nameLabel.substring(0, 64) : nameLabel;
    var labels = List.of(prefixLabel, nameLabel);
    payload.put(LABELS_NODE, labels);
    return payload;
  }

  public static QueryParameter getContainerQueryParameter(String tenantId) {
    return new QueryParameter(
        CONTAINER_FILTER_PARAMETER, BTP_TENANT_LABEL_QUERY.formatted(tenantId));
  }

  public static QueryParameter getExpandContainersAndCredentials() {
    return new QueryParameter(EXPAND_PARAMETER, EXPAND_CONTAINERS_AND_CREDENTIALS);
  }

  public static List<HanaTenant> getTenantsFromPayload(Map<String, Object> payload) {
    var hanaTenants = new ArrayList<HanaTenant>();
    if (payload.containsKey(DATA_NODE)) {
      asListMap(payload.get(DATA_NODE))
          .forEach(tenantData -> hanaTenants.add(new HanaTenant(tenantData)));
    }
    return hanaTenants;
  }

  public String getDatabaseId() {
    var dataBaseId = (String) get(SERVICE_INSTANCE_ID_NODE);
    if (StringUtils.isBlank(dataBaseId)) {
      return "";
    }
    return dataBaseId;
  }

  public Status getStatus() {
    return new Status(asMap(get(STATUS_NODE)));
  }

  public String getId() {
    if (!containsKey(ID_NODE)) {
      return "";
    }
    return (String) get(ID_NODE);
  }

  public List<Container> getContainers() {
    return containers;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    if (!super.equals(o)) {
      return false;
    }
    HanaTenant that = (HanaTenant) o;
    return Objects.equals(containers, that.containers);
  }

  @Override
  public int hashCode() {
    return Objects.hash(super.hashCode(), containers);
  }
}
