/*
 * © 2023-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.mt.impl;

import com.sap.cds.feature.mt.MtUtils;
import com.sap.cds.feature.mt.lib.subscription.HanaEncryptionTool;
import com.sap.cds.feature.mt.lib.subscription.HdiContainerManager;
import com.sap.cds.feature.mt.lib.subscription.InstanceLifecycleManager;
import com.sap.cds.feature.mt.lib.subscription.LiquibaseParameters;
import com.sap.cds.feature.mt.lib.subscription.PollingParameters;
import com.sap.cds.feature.mt.lib.subscription.ProvisioningService;
import com.sap.cds.feature.mt.lib.subscription.SecurityChecker;
import com.sap.cds.feature.mt.lib.subscription.ServiceSpecification;
import com.sap.cds.feature.mt.lib.subscription.Subscriber;
import com.sap.cds.feature.mt.lib.subscription.SubscriberBuilder;
import com.sap.cds.feature.mt.lib.subscription.SubscriberImpl;
import com.sap.cds.feature.mt.lib.subscription.exceptions.AuthorityError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.NotFound;
import com.sap.cds.feature.mt.lib.subscription.exceptions.NotSupported;
import com.sap.cds.feature.mt.lib.subscription.exceptions.ParameterError;
import com.sap.cds.feature.mt.lib.subscription.exits.Exits;
import com.sap.cds.feature.mt.lib.subscription.exits.SubscribeExit;
import com.sap.cds.feature.mt.lib.subscription.exits.UnSubscribeExit;
import com.sap.cds.feature.mt.lib.subscription.json.DeletePayload;
import com.sap.cds.feature.mt.lib.subscription.json.SubscriptionPayload;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.DeploymentService;
import com.sap.cds.services.mt.SubscribeEventContext;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.mt.UnsubscribeEventContext;
import com.sap.cds.services.mt.UpgradeEventContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.lib.tools.api.InstanceCreationOptions;
import com.sap.cds.services.utils.lib.tools.api.ResilienceConfig;
import com.sap.cds.services.utils.model.DynamicModelUtils;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.security.BasicCredentials;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** The default handler for subscription events. */
@ServiceName(DeploymentService.DEFAULT_NAME)
@SuppressWarnings({"deprecation", "removal"})
public class MtDeploymentServiceHandler implements EventHandler {

  private static final Logger logger = LoggerFactory.getLogger(MtDeploymentServiceHandler.class);
  public static final String ALL_TENANTS = "all";

  /** The subscriber that performs the actual tenant onboarding */
  protected final Subscriber subscriber;

  /** Thread local used to pass the instance manager options to the mt lib */
  protected final ThreadLocal<InstanceCreationOptions> options = new ThreadLocal<>();

  private final boolean useHanaX509;

  public MtDeploymentServiceHandler(
      InstanceLifecycleManager instanceLifecycleManager, CdsRuntime runtime) {
    try {
      MultiTenancy config = runtime.getEnvironment().getCdsProperties().getMultiTenancy();

      this.useHanaX509 = config.getDataSource().getX509().isEnabled();
      MtUtils mtUtils = new MtUtils(runtime);
      DynamicModelUtils dynamicModelUtils = new DynamicModelUtils(runtime);
      ResilienceConfig resilienceConfig = dynamicModelUtils.getResilienceConfig();
      List<HttpDestination> destinations = new ArrayList<>();

      ProvisioningService provisioningService = null;
      if (!StringUtils.isEmpty(mtUtils.getProvisioningServiceUrl())) {
        PollingParameters pollingParameters =
            PollingParameters.Builder.create()
                .interval(config.getProvisioning().getPollingInterval())
                .timeout(config.getProvisioning().getPollingTimeout())
                .build();

        provisioningService =
            new ProvisioningService(
                ServiceSpecification.Builder.create()
                    .resilienceConfig(resilienceConfig)
                    .polling(pollingParameters)
                    .build());

        destinations.add(
            dynamicModelUtils.createSidecarDestination(
                ProvisioningService.MTX_PROVISIONING_SERVICE_DESTINATION,
                mtUtils.getProvisioningServiceUrl()));
      }

      // legacy support for dynamic HDI deployer
      HdiContainerManager hdiContainerManager = null;
      if (!StringUtils.isEmpty(config.getDeployer().getUrl())) {
        PollingParameters pollingParameters =
            PollingParameters.Builder.create()
                .interval(Duration.ofSeconds(5))
                .timeout(config.getDeployer().getAsyncTimeout())
                .build();

        hdiContainerManager =
            new HdiContainerManager(
                ServiceSpecification.Builder.create()
                    .polling(pollingParameters)
                    .resilienceConfig(resilienceConfig)
                    .build(),
                null);

        destinations.add(
            DefaultHttpDestination.builder((config.getDeployer().getUrl()))
                .name(HdiContainerManager.HDI_DEPLOYER_DESTINATION)
                .basicCredentials(
                    new BasicCredentials(
                        config.getDeployer().getUser(), config.getDeployer().getPassword()))
                .build());
      }

      // TODO sync with other destinations
      DefaultDestinationLoader destinationLoader = new DefaultDestinationLoader();
      destinations.forEach(d -> destinationLoader.registerDestination(d));
      DestinationAccessor.prependDestinationLoader(destinationLoader);

      String encryptionMode = config.getDataSource().getHanaEncryptionMode();
      subscriber =
          SubscriberBuilder.create()
              .instanceLifecycleManager(instanceLifecycleManager)
              .hdiContainerManager(hdiContainerManager)
              .provisioningService(provisioningService)
              .exits(getExits())
              .securityChecker(getSecurityChecker())
              .hanaEncryptionMode(
                  encryptionMode != null
                      ? HanaEncryptionTool.DbEncryptionMode.valueOf(encryptionMode)
                      : null)
              .liquibaseParameters(
                  new LiquibaseParameters(
                      config.getLiquibase().getChangeLog(),
                      config.getLiquibase().getContexts(),
                      null))
              .baseUiUrl(config.getAppUi().getUrl())
              .urlSeparator(config.getAppUi().getTenantSeparator())
              .build();
    } catch (IllegalArgumentException | InternalError e) {
      throw new ErrorStatusException(CdsErrorStatuses.SUBSCRIBER_FAILED, e);
    }
  }

  private SecurityChecker getSecurityChecker() {
    return new SecurityChecker() {

      @Override
      public void checkSubscriptionAuthority() throws AuthorityError {
        // done in before handler
      }

      @Override
      public void checkInitDbAuthority() throws AuthorityError {
        // done in before handler
      }
    };
  }

  private Exits getExits() {
    return new Exits(
        new UnSubscribeExit() {
          @Override
          public Boolean onBeforeUnsubscribe(String tenantId, DeletePayload deletePayload) {
            return true;
          }

          @Override
          public Boolean onBeforeAsyncUnsubscribe(String tenantId, DeletePayload deletePayload) {
            return true;
          }
        },
        new SubscribeExit() {
          @Override
          public InstanceCreationOptions onBeforeSubscribe(
              String tenantId, SubscriptionPayload subscriptionPayload) throws InternalError {
            return options.get();
          }

          @Override
          public InstanceCreationOptions onBeforeAsyncSubscribe(
              String tenantId, SubscriptionPayload subscriptionPayload) throws InternalError {
            return options.get();
          }
        },
        null,
        null,
        null,
        false);
  }

  @On
  @HandlerOrder(OrderConstants.On.PRIORITY)
  protected void onSubscribe(SubscribeEventContext context) {
    try {
      options.set(buildInstanceCreationOptions(context));
      var payload = new SubscriptionPayload(context.getOptions());
      subscriber.subscribe(context.getTenant(), payload);
    } catch (InternalError e) {
      throw new ErrorStatusException(CdsErrorStatuses.SUBSCRIPTION_FAILED, context.getTenant(), e);
    } catch (ParameterError e) {
      throw new ErrorStatusException(ErrorStatuses.BAD_REQUEST, e);
    } catch (AuthorityError e) {
      throw new ErrorStatusException(ErrorStatuses.FORBIDDEN, e);
    } finally {
      options.remove();
    }
  }

  @On
  protected void onUnsubscribe(UnsubscribeEventContext context) {
    try {
      logger.info("Deleting subscription of tenant '{}'", context.getTenant());
      var payload = new DeletePayload(context.getOptions());
      subscriber.unsubscribe(context.getTenant(), payload);
    } catch (InternalError e) {
      throw new ErrorStatusException(
          CdsErrorStatuses.UNSUBSCRIPTION_FAILED, context.getTenant(), e);
    } catch (ParameterError e) {
      throw new ErrorStatusException(ErrorStatuses.BAD_REQUEST, e);
    } catch (AuthorityError e) {
      throw new ErrorStatusException(ErrorStatuses.FORBIDDEN, e);
    }
  }

  @On
  @HandlerOrder(OrderConstants.On.PRIORITY)
  protected void onUpgrade(UpgradeEventContext context) {
    var tenants = context.getTenants();
    if (tenants.size() == 1 && tenants.get(0).equals(ALL_TENANTS)) {
      tenants =
          context
              .getServiceCatalog()
              .getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME)
              .readTenants();
    }
    try {
      if (subscriber instanceof SubscriberImpl) {
        // HDI deployer
        subscriber.setupDbTables(tenants);
      } else {
        // Sidecar
        new AsyncSidecarUpgradeHelper(subscriber).deploy(tenants);
      }
    } catch (NotFound | InternalError e) {
      throw new ErrorStatusException(
          CdsErrorStatuses.DEPLOYMENT_FAILED, String.join(",", context.getTenants()), e);
    } catch (NotSupported e) {
      throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED, e);
    } catch (ParameterError e) {
      throw new ErrorStatusException(ErrorStatuses.BAD_REQUEST, e);
    } catch (AuthorityError e) {
      throw new ErrorStatusException(ErrorStatuses.FORBIDDEN, e);
    }
  }

  @Before
  @SuppressWarnings("unchecked")
  protected void beforeSubscribe(SubscribeEventContext context) {
    if (this.useHanaX509) {
      logger.debug(
          "Setting credential type to X509 for subscription of tenant '{}'", context.getTenant());

      if (context.getOptions() == null) {
        context.setOptions(new HashMap<>());
      }

      Map<String, Object> binding =
          (Map<String, Object>)
              context.getOptions().computeIfAbsent("bindingParameters", k -> new HashMap<>());
      binding.put("credential-type", "X509");
    }
  }

  @SuppressWarnings("unchecked")
  protected InstanceCreationOptions buildInstanceCreationOptions(SubscribeEventContext context) {
    InstanceCreationOptions ico = new InstanceCreationOptions();
    Object provisioning = context.getOptions().get("provisioningParameters");
    if (provisioning instanceof Map) {
      ico.withProvisioningParameters((Map<String, Object>) provisioning);
    }
    Object binding = context.getOptions().get("bindingParameters");
    if (binding instanceof Map) {
      ico.withBindingParameters((Map<String, Object>) binding);
    }
    return ico.getProvisioningParameters() != null || ico.getBindingParameters() != null
        ? ico
        : null;
  }
}
