/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Objects;
import com.sap.cds.Struct;
import com.sap.cds.integration.cloudsdk.rest.client.JsonRestClient;
import com.sap.cds.services.mt.SmsCallback;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.environment.ServiceBindingUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.environment.servicebinding.api.ServiceIdentifier;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultOAuth2PropertySupplier;
import com.sap.cloud.sdk.cloudplatform.connectivity.OAuth2ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.UnaryOperator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Implementation of the SMS REST client. */
public class SmsClient extends JsonRestClient {

  private static final Logger logger = LoggerFactory.getLogger(SmsClient.class);

  private static final int PAGE_SIZE = 500;

  private static final String SMS = "subscription-manager";

  public static final String BINDING_URL_KEY = "subscription_manager_url";

  public static final String SUBSCRIPTIONS_PATH = "/subscription-manager/v1/subscriptions";

  public static final String SUBSCRIPTION_STATE = "subscriptionState";

  private static final String KEY_SUBSCRIBER = "subscriber";
  private static final String KEY_ZONE_ID = "zoneId";
  private static final String KEY_APP_TID = "app_tid";
  private static final String KEY_SUBACCOUNT_SUBDOMAIN = "subaccountSubdomain";

  static {
    OAuth2ServiceBindingDestinationLoader.registerPropertySupplier(
        ServiceIdentifier.of(SMS), SmsClient.OAuth2Supplier::new);
  }

  private static class OAuth2Supplier extends DefaultOAuth2PropertySupplier {

    public OAuth2Supplier(ServiceBindingDestinationOptions options) {
      super(options, Collections.emptyList());
    }

    @Override
    public URI getServiceUri() {
      return getCredentialOrThrow(URI.class, BINDING_URL_KEY);
    }
  }

  public static Optional<ServiceBinding> findBinding(CdsRuntime runtime) {
    return runtime
        .getEnvironment()
        .getServiceBindings()
        .filter(b -> ServiceBindingUtils.matches(b, null, SMS))
        .findFirst();
  }

  @SuppressWarnings("unchecked")
  public static TenantInfo toTenantInfo(Map<String, Object> tenant) {
    TenantInfo info = Struct.access(tenant).as(TenantInfo.class);
    Map<String, Object> subscriber = (Map<String, Object>) info.get(KEY_SUBSCRIBER);

    String tenantId = (String) subscriber.get(KEY_APP_TID);
    // fallback to zoneId if app_tid does not exist, yet
    if (tenantId == null) {
      tenantId = (String) subscriber.get(KEY_ZONE_ID);
    }
    info.setTenant(tenantId);

    String subdomain = (String) subscriber.get(KEY_SUBACCOUNT_SUBDOMAIN);
    info.setSubdomain(subdomain);

    return info;
  }

  public SmsClient(ServiceBinding binding) {
    super(
        ServiceBindingDestinationOptions.forService(binding)
            .onBehalfOf(OnBehalfOf.TECHNICAL_USER_PROVIDER)
            .build());
  }

  public void callBackSms(SmsCallback payload, String callbackUrl) throws IOException {
    ObjectNode body = mapper.convertValue(payload, ObjectNode.class);
    logger.debug("Calling subscription manager");
    putRequestWithOnlyResponseCode(callbackUrl, body);
  }

  /**
   * Retrieves the tenant info of all subscribed tenants
   *
   * @param filter function to filter the tenants
   * @return tenants info
   * @throws IOException throws when any connection problems occur
   */
  public List<TenantInfo> getSubscribedTenants(UnaryOperator<TenantInfo> filter)
      throws IOException {
    List<TenantInfo> result = new ArrayList<>();
    boolean morePages = true;
    int skip = 0;

    while (morePages) {
      logger.debug(
          "Retrieving page {} of all tenants metadata from the Subscription Manager Service",
          (skip / PAGE_SIZE) + 1);

      String paginatedUrlPath = paginatedUrl(SUBSCRIPTIONS_PATH, skip);
      JsonNode jsonResponse = getRequest(paginatedUrlPath);
      ArrayNode resp = getNode(jsonResponse, "subscriptions");
      BooleanNode morePagesResp = getNode(jsonResponse, "morePages");

      resp.forEach(
          tenant -> {
            String state =
                tenant.get(SUBSCRIPTION_STATE) != null
                    ? tenant.get(SUBSCRIPTION_STATE).asText()
                    : null;
            if (Objects.equal(state, "SUBSCRIBED") || Objects.equal(state, "UPDATE_FAILED")) {
              TenantInfo tenantInfo =
                  toTenantInfo(mapper.convertValue(tenant, new TypeReference<>() {}));
              if (filter != null) {
                tenantInfo = filter.apply(tenantInfo);
              }
              result.add(tenantInfo);
            }
          });

      morePages = morePagesResp != null && morePagesResp.booleanValue();
      skip = skip + PAGE_SIZE;

      logger.debug(
          "More tenants metadata from the Subscription Manager Service available: {}",
          morePages ? "yes" : "no");
    }

    return result;
  }

  /**
   * Retrieves the tenant info of all subscribed tenants
   *
   * @return tenants info
   * @throws IOException throws when any connection problems occur
   */
  public List<TenantInfo> getSubscribedTenants() throws IOException {
    return getSubscribedTenants(null);
  }

  private static String paginatedUrl(String urlPath, int skip) {
    String url = urlPath + (urlPath.contains("?") ? "&" : "?");
    url = url + "$skip=" + skip + "&$top=" + PAGE_SIZE;
    return url;
  }

  @SuppressWarnings("unchecked")
  private static <N> N getNode(JsonNode jsonResponse, String keyName) {
    return (N) jsonResponse.get(keyName);
  }
}
