/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import java.util.Locale;
import java.util.function.Predicate;

public class FilterTenants {
  // used if no tenant id is provided for JPA
  public static final String MT_LIB_TENANT = "MT_LIB_TENANT";
  // the libs own tenants, one per DB, full name is MT_LIB_TENANT-<database id>
  public static final String MT_LIB_TENANT_PREFIX = "MT_LIB_TENANT-";
  // used by mtx/sidecar
  private static final String META = "__META__";
  private static final String TENANT = "TENANT";
  // used by old mt-lib
  private static final String DUMMY = "DUMMY";
  // migrated tenants from HAAS to HANA cloud
  private static final String MIGRATED = "_migrated";
  // used by sidecar to store for example jobs
  private static final String SIDECAR_TENANT = "t0";

  private FilterTenants() {}

  /**
   * Filter that returns true for "real" on-boarded tenants. Technical tenants used by mtx or the
   * lib's own tenant MT_LIB_TENANT are filtered out.
   *
   * @return Filter for on-boarded not technical tenants
   */
  static Predicate<String> realTenants() {
    return tenantId ->
        !(tenantId.startsWith(TENANT)
            || tenantId.equals(MT_LIB_TENANT)
            || tenantId.equals(DUMMY)
            || tenantId.equals(META)
            || tenantId.startsWith(MT_LIB_TENANT_PREFIX)
            || tenantId.toLowerCase(Locale.US).endsWith(MIGRATED)
            || tenantId.toLowerCase(Locale.US).equals(SIDECAR_TENANT));
  }

  // because of compatibility
  static Predicate<String> externalTenants() {
    return realTenants();
  }
}
