/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import java.time.Duration;
import java.time.Instant;

public class PollingParameters {

  /** The time waited between polling calls. */
  private Duration interval;

  /** Maximum waiting time for a remote operation */
  private Duration requestTimeout;

  private PollingParameters() {}

  public <E extends Exception> boolean pollUntil(Until<E> untilSupplier) throws E {
    var start = Instant.now();
    boolean first = true;
    do {
      boolean until = untilSupplier.calculateAndCheck(first);
      first = false;
      if (until || Duration.between(start, Instant.now()).compareTo(requestTimeout) > 0) {
        return until;
      }
      try {
        Thread.sleep(interval.toMillis());
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
        return false;
      }
    } while (true);
  }

  public Duration getInterval() {
    return interval;
  }

  public void setInterval(Duration interval) {
    this.interval = interval;
  }

  public Duration getRequestTimeout() {
    return requestTimeout;
  }

  public void setRequestTimeout(Duration requestTimeout) {
    this.requestTimeout = requestTimeout;
  }

  @FunctionalInterface
  public interface Until<E extends Exception> {

    public boolean calculateAndCheck(boolean first) throws E;
  }

  public static final class Builder {

    private Duration interval = Duration.ofSeconds(5);
    private Duration timeout = Duration.ofMinutes(10);

    private Builder() {}

    public static Builder create() {
      return new Builder();
    }

    public Builder interval(Duration interval) {
      this.interval = interval;
      return this;
    }

    public Builder timeout(Duration timeout) {
      this.timeout = timeout;
      return this;
    }

    public PollingParameters build() {
      PollingParameters pollingParameters = new PollingParameters();
      pollingParameters.setInterval(interval);
      pollingParameters.setRequestTimeout(timeout);
      return pollingParameters;
    }
  }
}
