/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription;

import com.sap.cds.feature.mt.lib.subscription.ServiceOperation.Status;
import com.sap.cds.feature.mt.lib.subscription.ServiceOperation.Type;
import com.sap.cds.services.utils.lib.tools.api.Cloner;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.lang3.StringUtils;

/** Service instance parameters */
public class ServiceInstance extends HashMap<String, Object> {
  public static final String ID = "id";
  public static final String NAME = "name";
  public static final String LABELS = "labels";
  public static final String TENANT_ID = "tenant_id";
  public static final String USABLE = "usable";
  private final ServiceOperation lastOperation;
  private List<ServiceBinding> bindings = new ArrayList<>();
  private ServiceBinding newestBinding = null;

  public ServiceInstance(Map<String, ?> m) {
    super(m);
    lastOperation = new ServiceOperation(m != null ? m : new HashMap<String, Object>());
  }

  @SuppressWarnings("unchecked")
  public ServiceInstance createCopy() {
    var serviceInstanceCopy = new ServiceInstance(Cloner.clone(this, Map.class));
    serviceInstanceCopy.setBindings(bindings);
    return serviceInstanceCopy;
  }

  public String getId() {
    return containsKey(ID) ? (String) get(ID) : "";
  }

  public boolean isReady() {
    return containsKey(USABLE) && (boolean) get(USABLE);
  }

  public String getName() {
    return containsKey(NAME) ? (String) get(NAME) : "";
  }

  public String getServicePlanId() {
    return (String) get("service_plan_id");
  }

  @SuppressWarnings("unchecked")
  public List<String> getTenants() {
    var labels = (Map<String, Object>) get(LABELS);
    if (labels == null) {
      return new ArrayList<>();
    }
    var tenants = (List<String>) labels.get(TENANT_ID);
    return tenants != null
        ? tenants.stream().filter(t -> !StringUtils.isBlank(t)).toList()
        : new ArrayList<>();
  }

  @SuppressWarnings("unchecked")
  public void insertTenant(String tenantId) {
    var labels =
        (Map<String, Object>) computeIfAbsent(LABELS, key -> new HashMap<String, Object>());
    var tenants = (List<String>) labels.computeIfAbsent(TENANT_ID, key -> new ArrayList<String>());
    tenants.add(tenantId);
  }

  @SuppressWarnings("unchecked")
  public void clearTenants() {
    var labels = (Map<String, Object>) get(LABELS);
    if (labels == null) {
      return;
    }
    var tenants = (List<String>) labels.get(TENANT_ID);
    if (tenants != null) {
      tenants.clear();
    }
  }

  public boolean hasTenant() {
    return !getTenants().isEmpty();
  }

  public Instant getCreatedAt() {
    return Instant.parse((String) get("created_at"));
  }

  public Instant getUpdatedAt() {
    return Instant.parse((String) get("updated_at"));
  }

  public String getUsable() {
    return (String) get(USABLE);
  }

  public ServiceOperation getLastOperation() {
    return lastOperation;
  }

  public List<ServiceBinding> getBindings() {
    return bindings;
  }

  public void setBindings(List<ServiceBinding> bindings) {
    this.bindings = bindings != null ? bindings : new ArrayList<>();
    final AtomicReference<ServiceBinding> newestBindingRef = new AtomicReference<>(null);
    this.bindings.stream()
        .filter(b -> b.getServiceInstanceId().equals(getId()))
        .filter(ServiceBinding::isUsable)
        .forEach(
            b -> {
              if (newestBindingRef.get() == null
                  || b.getCreatedAt().isAfter(newestBindingRef.get().getCreatedAt())) {
                newestBindingRef.set(b);
              }
            });
    if (newestBindingRef.get() == null) {
      this.newestBinding = null;
    } else {
      this.newestBinding = newestBindingRef.get();
    }
  }

  public Optional<ServiceBinding> getBinding() {
    return newestBinding != null ? Optional.of(newestBinding) : Optional.empty();
  }

  public boolean isUsable() {
    var status = lastOperation.getStatus();
    return isReady()
        && !(lastOperation.getType() == Type.CREATE
            && (status == Status.FAILED || status == Status.IN_PROGRESS));
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    if (!super.equals(o)) return false;
    ServiceInstance that = (ServiceInstance) o;
    return Objects.equals(bindings, that.bindings)
        && Objects.equals(newestBinding, that.newestBinding)
        && Objects.equals(lastOperation, that.lastOperation);
  }

  @Override
  public int hashCode() {
    return Objects.hash(super.hashCode(), bindings, newestBinding, lastOperation);
  }
}
