/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.subscription.hana.mt.service;

import static com.sap.cds.services.utils.lib.tools.impl.Wrap.wrap;
import static com.sap.cds.services.utils.lib.tools.impl.Wrap.wrapEmpty;
import static org.apache.http.HttpStatus.SC_ACCEPTED;
import static org.apache.http.HttpStatus.SC_BAD_GATEWAY;
import static org.apache.http.HttpStatus.SC_CONFLICT;
import static org.apache.http.HttpStatus.SC_GATEWAY_TIMEOUT;
import static org.apache.http.HttpStatus.SC_INTERNAL_SERVER_ERROR;
import static org.apache.http.HttpStatus.SC_NO_CONTENT;
import static org.apache.http.HttpStatus.SC_OK;
import static org.apache.http.HttpStatus.SC_SERVICE_UNAVAILABLE;

import com.sap.cds.feature.mt.lib.subscription.BindingParameters;
import com.sap.cds.feature.mt.lib.subscription.HanaAccess;
import com.sap.cds.feature.mt.lib.subscription.PollingParameters.Until;
import com.sap.cds.feature.mt.lib.subscription.ProvisioningParameters;
import com.sap.cds.feature.mt.lib.subscription.ServiceInstance;
import com.sap.cds.feature.mt.lib.subscription.ServiceSpecification;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.services.utils.lib.tools.api.QueryParameter;
import com.sap.cds.services.utils.lib.tools.api.ServiceCall;
import com.sap.cds.services.utils.lib.tools.api.ServiceEndpoint;
import com.sap.cds.services.utils.lib.tools.api.ServiceResponse;
import com.sap.cds.services.utils.lib.tools.exception.InternalException;
import com.sap.cds.services.utils.lib.tools.exception.ServiceException;
import com.sap.cds.services.utils.lib.tools.impl.Wrap;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.OAuth2Options;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HanaMtService implements HanaAccess {

  private static final Logger logger = LoggerFactory.getLogger(HanaMtService.class);
  private static final String TENANTS_ENDPOINT = "/tenants/v2/tenants";
  private static final String CONTAINERS_ENDPOINT = "/tenants/v2/tenants/%s/containers";
  private static final String CONTAINER_ENDPOINT = "/tenants/v2/tenants/%s/containers/%s";
  private static final String CREDENTIALS_ENDPOINT =
      "/tenants/v2/tenants/%s/containers/%s/credentials";
  private static final String CREDENTIAL_ENDPOINT =
      "/tenants/v2/tenants/%s/containers/%s/credentials/%s";
  private static final String UNEXPECTED_RETURN_CODE = "unexpected return code %d";
  private static final String BASEURL = "baseurl";
  private static final String SKIPTOKEN = "$skiptoken";
  private static final String LOCATION = "location";
  private static final String IF_MATCH = "If-Match";
  private static final String NO_ETAG_RETURNED = "No etag returned";
  private static final String HANA_TENANT_ID_IN_PROVISIONING = "hana_tenant_id";

  static {
    HanaMtPropertySupplier.initialize();
  }

  private final ServiceEndpoint hanaTenantsEndpoint;
  private final ServiceEndpoint containersEndpoint;
  private final ServiceEndpoint containerEndpoint;
  private final ServiceEndpoint credentialsEndpoint;
  private final ServiceEndpoint credentialEndpoint;
  private final ServiceSpecification serviceSpecification;
  private final com.sap.cloud.environment.servicebinding.api.ServiceBinding hanaMtBinding;
  private final AtomicBoolean dbIdsInitialized = new AtomicBoolean(false);
  private final Set<String> dbIds = ConcurrentHashMap.newKeySet();
  private final String serviceBindingName;
  private final String hanaTenantPrefix;

  public HanaMtService(
      com.sap.cloud.environment.servicebinding.api.ServiceBinding serviceBinding,
      ServiceSpecification serviceSpecification,
      Duration oauthTimeout,
      String hanaTenantPrefix)
      throws InternalError {
    this.serviceSpecification = serviceSpecification;
    this.hanaMtBinding = serviceBinding;
    this.serviceBindingName =
        serviceBinding
            .getName()
            .orElseThrow(() -> new InternalError("Service binding name is missing"));
    this.hanaTenantPrefix = hanaTenantPrefix;
    Set<Integer> retryCodes = new HashSet<>();
    retryCodes.add(SC_BAD_GATEWAY);
    retryCodes.add(SC_GATEWAY_TIMEOUT);
    retryCodes.add(SC_INTERNAL_SERVER_ERROR);
    retryCodes.add(SC_SERVICE_UNAVAILABLE);
    var destination = getDestination(serviceBinding, oauthTimeout);
    try {
      hanaTenantsEndpoint =
          createEndpoint(
              destination,
              TENANTS_ENDPOINT,
              new HashSet<>(List.of(SC_OK, SC_ACCEPTED, SC_NO_CONTENT)),
              retryCodes);
      containersEndpoint =
          createEndpoint(
              destination,
              CONTAINERS_ENDPOINT,
              new HashSet<>(List.of(SC_OK, SC_ACCEPTED, SC_NO_CONTENT, SC_CONFLICT)),
              retryCodes);
      containerEndpoint =
          createEndpoint(
              destination,
              CONTAINER_ENDPOINT,
              new HashSet<>(List.of(SC_OK, SC_ACCEPTED, SC_NO_CONTENT, SC_CONFLICT)),
              retryCodes);
      credentialsEndpoint =
          createEndpoint(
              destination,
              CREDENTIALS_ENDPOINT,
              new HashSet<>(List.of(SC_OK, SC_ACCEPTED, SC_NO_CONTENT, SC_CONFLICT)),
              retryCodes);
      credentialEndpoint =
          createEndpoint(
              destination,
              CREDENTIAL_ENDPOINT,
              new HashSet<>(List.of(SC_OK, SC_ACCEPTED, SC_NO_CONTENT, SC_CONFLICT)),
              retryCodes);
    } catch (InternalException e) {
      throw new InternalError(e);
    }
  }

  @Override
  public Optional<ServiceInstance> getInstance(String tenantId, boolean forceCacheUpdate)
      throws InternalError {
    checkTenantId(tenantId);
    var hanaTenantOpt = getOneHanaTenantViaBtpTenantId(tenantId);
    if (hanaTenantOpt.isEmpty()) {
      return Optional.empty();
    }
    var hanaTenant = hanaTenantOpt.get();
    dbIds.add(hanaTenant.getDatabaseId());
    var containers =
        hanaTenant.getContainers().stream().filter(c -> tenantId.equals(c.getBtpTenant())).toList();
    if (containers.size() > 1) {
      throw new InternalError(
          "Multiple HDI containers found for BTP tenant id {}".formatted(tenantId));
    }
    if (containers.isEmpty()) {
      logger.debug("No container found for HANA tenant {}", hanaTenant.getId());
      return Optional.empty();
    }
    var container = containers.get(0);
    logger.debug(
        "Container {} read with {} credentials",
        container.getId(),
        container.getCredentials().size());
    container
        .getCredentials()
        .forEach(
            c ->
                logger.debug(
                    "user = {} ,  hdiUser = {} , schema = {}",
                    c.getUser(),
                    c.getHdiUser(),
                    c.getSchema()));
    return Optional.of(containers.get(0).toServiceInstance());
  }

  /*
   * This method cannot be implemented with the HANA multi tenant service v2 because of rate limits:
   * Determining a BTP tenant, requires a expand down to HDI container level. This will be allowed only
   * for up to 10 tenants. It is currently also a slow operation that requires 30 minutes for 10000 tenants.
   *
   */
  @Override
  public List<TenantInfo> getAllTenants(boolean forceCacheUpdate) throws InternalError {
    throw new InternalError(
        "Method getAllTenants isn't supported for HANA multi tenant service v2");
  }

  @Override
  public Set<String> getDatabaseIds(boolean forceCacheUpdate) throws InternalError {
    if (forceCacheUpdate || !dbIdsInitialized.get()) {
      synchronized (this) {
        if (forceCacheUpdate || !dbIdsInitialized.get()) {
          dbIds.addAll(
              readHanaTenants(null, null, false).stream()
                  .map(HanaTenant::getDatabaseId)
                  .filter(StringUtils::isNotBlank)
                  .collect(Collectors.toSet()));
          dbIdsInitialized.set(true);
        }
      }
    }
    return new HashSet<>(dbIds);
  }

  /*
    Please note, a HANA tenant is never deleted automatically as it can be reused across
    microservices.
  */
  @Override
  public void deleteInstance(String tenantId) throws InternalError {
    checkTenantId(tenantId);
    var hanaTenantOpt = getOneHanaTenantViaBtpTenantId(tenantId);
    if (hanaTenantOpt.isEmpty()) {
      return;
    }
    var hanaTenant = hanaTenantOpt.get();
    for (var container : hanaTenant.getContainers()) {
      for (var credential : container.getCredentials()) {
        deleteCredential(hanaTenant.getId(), container.getId(), credential.getId());
      }
      waitForCredentialDeletion(container, hanaTenant);
      deleteContainer(hanaTenant.getId(), container.getId());
      checkThatContainersWereDeleted(hanaTenant);
    }
  }

  @Override
  public ServiceInstance createInstance(
      String tenantId,
      ProvisioningParameters provisioningParameters,
      BindingParameters bindingParameters)
      throws InternalError {
    logger.debug("Create a new service instance for btp tenant id {}", tenantId);
    var polling = serviceSpecification.getPolling();
    var hanaTenantWrap = wrapEmpty(HanaTenant.class);
    logger.debug("Poll until HANA tenant is created");
    polling.pollUntil(
        untilHanaTenantIsReadOrCreated(tenantId, hanaTenantWrap, provisioningParameters));
    var hanaTenant =
        hanaTenantWrap.orElseThrow(() -> new InternalError("HANA tenant couldn't be created"));
    dbIds.add(hanaTenant.getDatabaseId());
    var containerWithEtagWrap = wrapEmpty(ContainerWithEtag.class);
    logger.debug("Poll until container is created for HANA tenant {}", hanaTenant.getId());
    polling.pollUntil(
        untilContainerIsReadOrCreated(tenantId, hanaTenant.getId(), containerWithEtagWrap));
    var container =
        containerWithEtagWrap
            .get()
            .container()
            .orElseThrow(() -> new InternalError("HDI container couldn't be created"));
    var credentialWithEtagWrap = wrapEmpty(CredentialWithEtag.class);
    logger.debug(
        "Poll until credential is created for HANA tenant {} and container {}",
        hanaTenant.getId(),
        container.getId());
    polling.pollUntil(
        untilCredentialIsReady(hanaTenant.getId(), container.getId(), credentialWithEtagWrap));
    var credential =
        credentialWithEtagWrap
            .get()
            .credential()
            .orElseThrow(() -> new InternalError("Credential couldn't be created"));

    container.insertCredentials(List.of(credential));
    return container.toServiceInstance();
  }

  @Override
  public void clearCache() {
    dbIds.clear();
  }

  public String getServiceBindingName() {
    return serviceBindingName;
  }

  private void waitForCredentialDeletion(Container container, HanaTenant hanaTenant)
      throws InternalError {
    var credentialsSizeWrap = wrapEmpty(Integer.class);
    serviceSpecification
        .getPolling()
        .pollUntil(
            (first) -> {
              try {
                var credentials =
                    readCredentials(hanaTenant.getId(), container.getId()).credentials();
                credentialsSizeWrap.set(credentials.size());
                return credentials.isEmpty();
              } catch (InternalError e) {
                logger.error("An exception occurred during credential read", e);
                return true;
              }
            });
    if (credentialsSizeWrap.isEmpty() || credentialsSizeWrap.get() != 0) {
      throw new InternalError("Not all credentials could be deleted");
    }
  }

  private @NotNull Until<InternalError> untilHanaTenantIsReadOrCreated(
      String tenantId,
      Wrap<HanaTenant> hanaTenantWrap,
      ProvisioningParameters provisioningParameters) {
    var hanaTenantIdWrap = wrapEmpty(String.class);
    return (first) -> {
      if (first) {
        var hanaTenants = readHanaTenants(null, tenantId, true);
        if (hanaTenants.size() > 1) {
          throw new InternalError(
              "Multiple HANA tenants found for tenant id %s".formatted(tenantId));
        }
        if (!hanaTenants.isEmpty()) {
          hanaTenantIdWrap.set(hanaTenants.get(0).getId());
          hanaTenantWrap.set(hanaTenants.get(0));
        }
      } else {
        var hanaTenantOpt = readHanaTenant(hanaTenantIdWrap.get(), false);
        if (hanaTenantOpt.isPresent()) {
          hanaTenantWrap.set(hanaTenantOpt.get());
        }
      }
      if (hanaTenantIdWrap.isEmpty()) {
        hanaTenantIdWrap.set(createHanaTenant(tenantId, provisioningParameters));
      }
      return hanaTenantWrap.isPresent() && hanaTenantWrap.get().getStatus().isReady();
    };
  }

  private @NotNull Until<InternalError> untilContainerIsReadOrCreated(
      String tenantId, String hanaTenantId, Wrap<ContainerWithEtag> containerWithEtagWrap) {
    var createRequestedWrap = wrap(false);
    return (first) -> {
      containerWithEtagWrap.set(readOneContainer(hanaTenantId, tenantId, false));
      if (containerWithEtagWrap.get().container().isEmpty()
          && !Boolean.TRUE.equals(createRequestedWrap.get())) {
        createRequestedWrap.set(true);
        var responseCode =
            createContainer(hanaTenantId, tenantId, containerWithEtagWrap.get().etag())
                .responseCode();
        if (responseCode == SC_CONFLICT) {
          // Create failed because another task is changing containers
          createRequestedWrap.set(false);
        }
      }
      // calculate until boolean
      return containerWithEtagWrap.get().container().isPresent()
          && containerWithEtagWrap.get().container().get().getStatus().isReady();
    };
  }

  private @NotNull Until<InternalError> untilCredentialIsReady(
      String hanaTenantId, String containerId, Wrap<CredentialWithEtag> credentialWithEtagWrap) {
    var createRequestedWrap = wrap(false);
    return (first) -> {
      logger.debug("Repeat util credential for container {} is ready", containerId);
      var credentialWithEtag = readNewestWorkingCredential(hanaTenantId, containerId);
      credentialWithEtagWrap.set(credentialWithEtag);
      if (credentialWithEtag.credential().isEmpty()
          && !Boolean.TRUE.equals(createRequestedWrap.get())) {
        logger.debug("No credential found: Create a new one.");
        createRequestedWrap.set(true);
        var responseCode =
            createCredential(hanaTenantId, containerId, credentialWithEtagWrap.get().etag())
                .responseCode();
        if (responseCode == SC_CONFLICT) {
          // Create failed because another task is changing containers
          createRequestedWrap.set(false);
        }
      }
      // calculate until boolean
      logger.debug(
          "Credential is present = {}, Credential is valid = {}",
          credentialWithEtag.credential().isPresent(),
          credentialWithEtag.credential().isPresent()
              ? credentialWithEtag.credential().get().isValid()
              : false);
      return credentialWithEtag.credential().isPresent()
          && credentialWithEtag.credential().get().isValid();
    };
  }

  private ContainerWithEtag readOneContainer(
      String hanaTenantId, String tenantId, boolean withExpand) throws InternalError {
    var containersWithEtag = readContainers(hanaTenantId, withExpand);
    var containers = containersWithEtag.containers();
    logger.debug("For HANA tenant {}, {} containers were read", hanaTenantId, containers.size());
    if (containers.isEmpty()) {
      return new ContainerWithEtag(Optional.empty(), containersWithEtag.etag());
    }
    if (containers.size() != 1) {
      throw new InternalError(
          "Several HDI containers exist for Hana tenant %s ".formatted(hanaTenantId));
    }
    var container = containers.get(0);
    if (!container.getBtpTenant().equals(tenantId)) {
      throw new InternalError(
          "Container is assigned to the wrong tenant id %s".formatted(container.getBtpTenant()));
    }
    if (containersWithEtag.etag().isBlank()) {
      throw new InternalError("Container etag is blank");
    }
    return new ContainerWithEtag(Optional.of(container), containersWithEtag.etag());
  }

  private CredentialWithEtag readNewestWorkingCredential(String hanaTenantId, String containerId)
      throws InternalError {
    var credentialsWithEtag = readCredentials(hanaTenantId, containerId);
    var credentials = credentialsWithEtag.credentials();
    if (credentials.isEmpty()) {
      return new CredentialWithEtag(Optional.empty(), credentialsWithEtag.etag());
    }
    if (credentialsWithEtag.etag().isBlank()) {
      throw new InternalError("Container etag is blank");
    }
    // take the last as newest, as no timestamp is included
    return new CredentialWithEtag(
        credentials.stream().filter(Credential::isValid).reduce((a, b) -> b),
        credentialsWithEtag.etag());
  }

  private ServiceEndpoint createEndpoint(
      HttpDestination destination,
      String path,
      Set<Integer> expectedResponseCodes,
      Set<Integer> retryCodes)
      throws InternalException {
    return ServiceEndpoint.create()
        .destination(destination)
        .path(path)
        .returnCodeChecker(
            c -> {
              if (!expectedResponseCodes.contains(c)) {
                return new InternalError(UNEXPECTED_RETURN_CODE.formatted(c));
              }
              return null;
            })
        .retry()
        .forReturnCodes(retryCodes)
        .config(serviceSpecification.getResilienceConfig())
        .end();
  }

  private List<HanaTenant> readHanaTenants(String skipToken, String tenantId, boolean withExpand)
      throws InternalError {
    logger.debug(
        "Determine Hana tenants for BTP tenant id {} and skipToken {}", tenantId, skipToken);
    try {
      var queryParameters = new ArrayList<QueryParameter>();
      if (StringUtils.isNotBlank(tenantId)) {
        queryParameters.add(HanaTenant.getContainerQueryParameter(tenantId));
      }
      if (withExpand) {
        queryParameters.add(HanaTenant.getExpandContainersAndCredentials());
      }
      if (StringUtils.isNotBlank(skipToken)) {
        queryParameters.add(new QueryParameter(SKIPTOKEN, skipToken));
      }
      var getHanaTenants =
          hanaTenantsEndpoint
              .createServiceCall()
              .http()
              .get()
              .withoutPayload()
              .noPathParameter()
              .query(queryParameters)
              .end();

      var startTime = Instant.now();
      ServiceResponse<Map> response = getHanaTenants.execute(Map.class);
      logger.debug(
          "Determination of HANA tenants for BTP tenant {} took {} ms, expand active={}",
          tenantId,
          Duration.between(startTime, Instant.now()).toMillis(),
          withExpand);

      var payload = response.getPayload().orElse(new HashMap<>());
      skipToken = (String) payload.get(SKIPTOKEN);
      var hanaTenants = HanaTenant.getTenantsFromPayload(payload);
      if (StringUtils.isNotBlank(skipToken)) {
        hanaTenants.addAll(readHanaTenants(skipToken, null, withExpand));
      }
      logger.debug("The number of found Hana tenants is {}", hanaTenants.size());
      return hanaTenants;
    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private ContainersWithEtag readContainers(String hanaTenantId, boolean withExpand)
      throws InternalError {
    try {
      var queryParameters = new ArrayList<QueryParameter>();
      if (withExpand) {
        queryParameters.add(Container.getExpandCredentials());
      }
      var getContainers =
          containersEndpoint
              .createServiceCall()
              .http()
              .get()
              .withoutPayload()
              .pathParameter(hanaTenantId)
              .query(queryParameters)
              .end();

      var startTime = Instant.now();
      ServiceResponse<Map> response = getContainers.execute(Map.class);
      logger.debug(
          "Determination of HDI containers for HANA tenant {} took {} ms, expand active={}",
          hanaTenantId,
          Duration.between(startTime, Instant.now()).toMillis(),
          withExpand);

      var payload = response.getPayload().orElse(new HashMap<>());
      var etag = response.getETag().orElseThrow(() -> new InternalError(NO_ETAG_RETURNED));
      if (etag.isBlank()) {
        throw new InternalError(NO_ETAG_RETURNED);
      }
      var containers = Container.getContainersFromPayload(payload);
      return new ContainersWithEtag(containers, etag);
    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private CredentialsWithEtag readCredentials(String hanaTenantId, String containerId)
      throws InternalError {
    try {
      var getCredentials =
          credentialsEndpoint
              .createServiceCall()
              .http()
              .get()
              .withoutPayload()
              .pathParameter(hanaTenantId, containerId)
              .noQuery()
              .end();

      var startTime = Instant.now();
      ServiceResponse<Map> response = getCredentials.execute(Map.class);
      logger.debug(
          "Determination of credentials for HANA tenant {} and container id={}, took {}ms",
          hanaTenantId,
          containerId,
          Duration.between(startTime, Instant.now()).toMillis());

      var payload = response.getPayload().orElse(new HashMap<>());
      var etag = response.getETag().orElseThrow(() -> new InternalError(NO_ETAG_RETURNED));
      if (etag.isBlank()) {
        throw new InternalError(NO_ETAG_RETURNED);
      }
      return new CredentialsWithEtag(Credential.getCredentialsFromPayload(payload), etag);
    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private void deleteCredential(String hanaTenantId, String containerId, String credentialId)
      throws InternalError {
    try {
      var deleteCredential =
          credentialEndpoint
              .createServiceCall()
              .http()
              .delete()
              .withoutPayload()
              .pathParameter(hanaTenantId, containerId, credentialId)
              .noQuery()
              .end();

      var startTime = Instant.now();
      deleteCredential.execute(Map.class);
      logger.debug(
          "Deletion of credentials with id {} took {}ms",
          containerId,
          Duration.between(startTime, Instant.now()).toMillis());

    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private void deleteContainer(String hanaTenantId, String containerId) throws InternalError {
    try {
      var deleteContainer =
          containerEndpoint
              .createServiceCall()
              .http()
              .delete()
              .withoutPayload()
              .pathParameter(hanaTenantId, containerId)
              .noQuery()
              .end();

      var startTime = Instant.now();
      deleteContainer.execute(Map.class);
      logger.debug(
          "Deletion of container with id {} took {}ms",
          containerId,
          Duration.between(startTime, Instant.now()).toMillis());

    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private Optional<HanaTenant> readHanaTenant(String hanaTenantId, boolean withExpand)
      throws InternalError {
    try {
      var queryParameters = new ArrayList<QueryParameter>();
      if (withExpand) {
        queryParameters.add(HanaTenant.getExpandContainersAndCredentials());
      }
      var getHanaTenant =
          hanaTenantsEndpoint
              .createServiceCall()
              .http()
              .get()
              .withoutPayload()
              .pathParameter(hanaTenantId)
              .query(queryParameters)
              .end();
      var response = getHanaTenant.execute(Map.class);
      return Optional.of(new HanaTenant(response.getPayload().orElse(new HashMap<>())));
    } catch (InternalException | ServiceException e) {
      logger.error("Could read HANA tenant %s".formatted(hanaTenantId), e);
      throw serviceErrorHandling(e);
    }
  }

  private HttpDestination getDestination(
      com.sap.cloud.environment.servicebinding.api.ServiceBinding binding, Duration oauthTimeout)
      throws InternalError {
    logger.debug("Create destination for Hana Mt service");
    if (StringUtils.isBlank((String) binding.getCredentials().get(BASEURL))) {
      throw new InternalError("Hana Multitenancy Service url is missing");
    }
    logger.debug("Hana API url is {}", binding.getCredentials().get(BASEURL));
    if (binding.getName().isEmpty() || StringUtils.isBlank(binding.getName().get())) {
      throw new InternalError("Service binding name is missing");
    }
    logger.debug("Hana API binding name is {}", binding.getName().get());
    if (binding.getServiceName().isEmpty() || StringUtils.isBlank(binding.getServiceName().get())) {
      throw new InternalError("Service name is missing");
    }
    var timeLimiterConfiguration =
        ResilienceConfiguration.TimeLimiterConfiguration.of(
            oauthTimeout != null ? oauthTimeout : Duration.ofSeconds(30));
    var destination =
        ServiceBindingDestinationLoader.defaultLoaderChain()
            .getDestination(
                ServiceBindingDestinationOptions.forService(binding)
                    .withOption(OAuth2Options.TokenRetrievalTimeout.of(timeLimiterConfiguration))
                    .onBehalfOf(OnBehalfOf.TECHNICAL_USER_PROVIDER)
                    .build());
    logger.debug("Hana Mt destination uri is {}", destination.getUri());
    return destination;
  }

  private void checkThatContainersWereDeleted(HanaTenant hanaTenant) throws InternalError {
    var containersSizeWrap = wrapEmpty(Integer.class);
    serviceSpecification
        .getPolling()
        .pollUntil(
            (first) -> {
              try {
                var containers = readContainers(hanaTenant.getId(), false).containers();
                containersSizeWrap.set(containers.size());
                return containers.isEmpty();
              } catch (InternalError e) {
                logger.error("An exception occurred during container deletion", e);
                return true;
              }
            });
    if (containersSizeWrap.isEmpty() || containersSizeWrap.get() != 0) {
      throw new InternalError("Not all HDI containers could be deleted");
    }
  }

  private InternalError serviceErrorHandling(Exception e) {
    if (e.getCause() instanceof InternalError internalError) {
      return internalError;
    }
    return new InternalError(e);
  }

  private void checkTenantId(String tenantId) throws InternalError {
    if (StringUtils.isBlank(tenantId)) {
      throw new InternalError("Tenant id is blank or empty");
    }
  }

  private Optional<HanaTenant> getOneHanaTenantViaBtpTenantId(String tenantId)
      throws InternalError {
    logger.debug("Read one HANA tenants via BTP tenant id {}", tenantId);
    var tenants = readHanaTenants(null, tenantId, true);
    if (tenants.isEmpty()) {
      return Optional.empty();
    }
    if (tenants.size() != 1) {
      throw new InternalError(
          "Multiple HANA tenants found for BTP tenant id %s".formatted(tenantId));
    }
    logger.debug("HANA tenant {} was found for BTP tenant {}", tenants.get(0).getId(), tenantId);
    return Optional.of(tenants.get(0));
  }

  private String createHanaTenant(String tenantId, ProvisioningParameters provisioningParameters)
      throws InternalError {
    var hanaTenantIdRecord =
        createHanaTenantId(
            (String) provisioningParameters.get(HANA_TENANT_ID_IN_PROVISIONING),
            hanaTenantPrefix,
            tenantId);
    logger.debug(
        "Create a new HANA tenant {} for BTP tenant id {}",
        hanaTenantIdRecord.hanaTenantId(),
        tenantId);

    try {
      ServiceCall createHanaTenant;
      var payload =
          HanaTenant.createCreatePayload(
              tenantId,
              provisioningParameters,
              hanaTenantIdRecord.prefix(),
              hanaTenantIdRecord.name());
      createHanaTenant =
          hanaTenantsEndpoint
              .createServiceCall()
              .http()
              .put()
              .payload(payload)
              .pathParameter(hanaTenantIdRecord.hanaTenantId())
              .query(null)
              .end();

      var startTime = Instant.now();
      var response = createHanaTenant.execute(Void.class);
      logger.debug(
          "Creation of HANA tenant for BTP tenant {} took {}ms",
          tenantId,
          Duration.between(startTime, Instant.now()).toMillis());

      var locationOpt =
          Arrays.stream(response.getHeaders())
              .filter(h -> h.getName().equalsIgnoreCase(LOCATION))
              .map(Header::getValue)
              .findFirst();
      if (locationOpt.isEmpty() || !locationOpt.get().contains(TENANTS_ENDPOINT + "/")) {
        throw new InternalError("No location returned");
      }
      return hanaTenantIdRecord.hanaTenantId();
    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private HanaTenantIdAndPrefixAndName createHanaTenantId(
      String externalHanaTenantId, String prefix, String tenantId) throws InternalError {
    if (StringUtils.isNotBlank(externalHanaTenantId)) {
      checkUuid(externalHanaTenantId);
      return new HanaTenantIdAndPrefixAndName(externalHanaTenantId, "", "");
    }
    if (StringUtils.isNotBlank(prefix)) {
      return new HanaTenantIdAndPrefixAndName(
          hashGuidFromString("%s-%s".formatted(prefix, tenantId)), prefix, tenantId);
    }
    throw new InternalError("Neither prefix nor explicit Hana tenant id specified");
    /* currently not possible in mtx sidecar, therefore alos not active in Java */
    /* var hanaMtServiceInstanceGuid =
        (String)
            hanaMtBinding
                .get("instance_guid")
                .orElseThrow(
                    () ->
                        new InternalError(
                            "No HANA instance id, prefix or service instance guid for HANA multi tenant service available"));
    checkUuid(hanaMtServiceInstanceGuid);
    return new HanaTenantIdAndPrefixAndName(
        hashGuidFromString("%s-%s".formatted(hanaMtServiceInstanceGuid, tenantId)),
        hanaMtServiceInstanceGuid,
        tenantId); */
  }

  private static void checkUuid(Object uuid) throws InternalError {
    try {
      UUID.fromString((String) uuid);
    } catch (IllegalArgumentException e) {
      throw new InternalError("Value %s is not an UUID".formatted(uuid));
    }
  }

  private ContainerIdWithResponseCode createContainer(
      String hanaTenantId, String tenantId, String etag) throws InternalError {
    var payload = Container.createCreatePayload(tenantId);
    try {
      ServiceCall createContainer;
      createContainer =
          containersEndpoint
              .createServiceCall()
              .http()
              .post()
              .payload(payload)
              .pathParameter(hanaTenantId)
              .query(null)
              .insertHeaderFields(Collections.singletonMap(IF_MATCH, etag))
              .end();

      var startTime = Instant.now();
      var response = createContainer.execute(Void.class);
      logger.debug(
          "Creation of HDI container for BTP tenant {} took {}ms",
          tenantId,
          Duration.between(startTime, Instant.now()).toMillis());

      var locationOpt =
          Arrays.stream(response.getHeaders())
              .filter(h -> h.getName().equalsIgnoreCase(LOCATION))
              .map(Header::getValue)
              .findFirst();
      var containerPath = CONTAINERS_ENDPOINT.formatted(hanaTenantId) + "/";
      if (locationOpt.isEmpty() || !locationOpt.get().contains(containerPath)) {
        throw new InternalError("No location returned");
      }
      String[] parts = locationOpt.get().split(containerPath);
      if (parts.length < 2 || parts[1].isEmpty()) {
        throw new InternalError("Container id is missing in the location");
      }
      return new ContainerIdWithResponseCode(parts[1], response.getHttpStatusCode());
    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private CredentialIdWithResponseCode createCredential(
      String hanaTenantId, String containerId, String etag) throws InternalError {
    var payload = Credential.createCreatePayload();
    try {
      ServiceCall createCredential;
      createCredential =
          credentialsEndpoint
              .createServiceCall()
              .http()
              .post()
              .payload(payload)
              .pathParameter(hanaTenantId, containerId)
              .noQuery()
              .insertHeaderFields(Collections.singletonMap(IF_MATCH, etag))
              .end();

      var startTime = Instant.now();
      var response = createCredential.execute(Void.class);
      logger.debug(
          "Creation of credentials for container with id {} took {}ms",
          containerId,
          Duration.between(startTime, Instant.now()).toMillis());

      var locationOpt =
          Arrays.stream(response.getHeaders())
              .filter(h -> h.getName().equalsIgnoreCase(LOCATION))
              .map(Header::getValue)
              .findFirst();
      var credentialPath = CREDENTIALS_ENDPOINT.formatted(hanaTenantId, containerId) + "/";
      if (locationOpt.isEmpty() || !locationOpt.get().contains(credentialPath)) {
        throw new InternalError("No location returned");
      }
      String[] parts = locationOpt.get().split(credentialPath);
      if (parts.length < 2 || parts[1].isEmpty()) {
        throw new InternalError("Credential id is missing in the location");
      }
      return new CredentialIdWithResponseCode(parts[1], response.getHttpStatusCode());
    } catch (InternalException | ServiceException e) {
      throw serviceErrorHandling(e);
    }
  }

  private String hashGuidFromString(String input) throws InternalError {
    try {
      MessageDigest md = MessageDigest.getInstance("SHA-256");
      byte[] hash = md.digest(input.getBytes("UTF-8"));
      ByteBuffer bb = ByteBuffer.wrap(hash);
      long mostSigBits = bb.getLong();
      long leastSigBits = bb.getLong();
      return (new UUID(mostSigBits, leastSigBits)).toString();
    } catch (Exception e) {
      throw new InternalError("Could not create a guid");
    }
  }

  private record HanaTenantIdAndPrefixAndName(String hanaTenantId, String prefix, String name) {}
}
