/*
 * © 2020-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.mt.impl;

import com.sap.cds.feature.mt.MtUtils;
import com.sap.cds.feature.mt.SaasClient;
import com.sap.cds.feature.mt.SmsClient;
import com.sap.cds.feature.mt.lib.subscription.InstanceLifecycleManager;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.mt.ExtensibilityService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.model.DynamicModelUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

public class MtDeploymentServiceConfiguration implements CdsRuntimeConfiguration {

  @Override
  public void services(CdsRuntimeConfigurer configurer) {
    // isModelProviderEnabled() doesn't fully guarantee extensibility usage (e.g. might only be
    // feature toggles usage)
    if (new DynamicModelUtils(configurer.getCdsRuntime()).isModelProviderEnabled()) {
      configurer.service(new ExtensibilityServiceImpl(ExtensibilityService.DEFAULT_NAME));
    }
  }

  @Override
  public void eventHandlers(CdsRuntimeConfigurer configurer) {
    CdsRuntime cdsRuntime = configurer.getCdsRuntime();

    MtUtils mtUtils = new MtUtils(cdsRuntime);
    InstanceLifecycleManager ilm = mtUtils.createDefaultInstanceLifecycleManager();

    ServiceBinding saasBinding = SaasClient.findBinding(cdsRuntime).orElse(null);
    ServiceBinding smsBinding = SmsClient.findBinding(cdsRuntime).orElse(null);

    if (saasBinding != null || smsBinding != null) {
      configurer.eventHandler(MtTenantProviderSaasOrSmsHandler.create(saasBinding, smsBinding));
    }

    if (ilm != null) {
      // adding this provider as well. SaaS handler is taken with priority based on the handler
      // order
      configurer.eventHandler(new MtTenantProviderHandler(ilm));
    }

    if (mtUtils.requiresSubscription()) {
      configurer.eventHandler(new MtDeploymentServiceHandler(ilm, cdsRuntime));

      // inform outbox about mt enablement, as persistent outbox is not usable for provider tenant
      // on tenant-aware databases
      cdsRuntime
          .getEnvironment()
          .getCdsProperties()
          .getOutbox()
          .getProviderTenant()
          .setEnabled(false);
    }

    // Audit log tenant on/off boarding
    CdsProperties.AuditLog.StandardEvents standardEvents =
        configurer
            .getCdsRuntime()
            .getEnvironment()
            .getCdsProperties()
            .getAuditLog()
            .getStandardEvents();
    if (standardEvents.isEnabled() && standardEvents.getTenantLifecycleEvents().isEnabled()) {
      AuditLogService als =
          configurer
              .getCdsRuntime()
              .getServiceCatalog()
              .getService(AuditLogService.class, AuditLogService.DEFAULT_NAME);
      configurer.eventHandler(new TenantSubscriptionAuditLogHandler(als));
    }

    CdsProperties.MultiTenancy.Dependencies dependencies =
        configurer
            .getCdsRuntime()
            .getEnvironment()
            .getCdsProperties()
            .getMultiTenancy()
            .getDependencies();
    if (dependencies.isPortal()) {
      configurer.eventHandler(
          new SubscriptionDependencyMtHandler(
              cdsRuntime.getEnvironment().getServiceBindings(),
              SubscriptionDependencyMtHandler.PORTAL));
    }

    if (dependencies.isHtml5AppsRepo()) {
      configurer.eventHandler(
          new SubscriptionDependencyMtHandler(
              cdsRuntime.getEnvironment().getServiceBindings(),
              SubscriptionDependencyMtHandler.HTML5_APPS_REPO));
    }

    configurer.eventHandler(new AppUiUrlEventHandler(cdsRuntime));
  }
}
