/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.mt.impl;

import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.DependenciesEventContext;
import com.sap.cds.services.mt.DeploymentService;
import com.sap.cds.services.utils.environment.ServiceBindingUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Generic implementation of the multi tenancy subscription handler which is responsible for
 * providing the dependencies to a given bound service. The handler should be reusable for services
 * that provide the `xsappname` in the binding under a path `uaa.xsappname`.
 */
@ServiceName(DeploymentService.DEFAULT_NAME)
public class SubscriptionDependencyMtHandler implements EventHandler {

  private static final Logger logger =
      LoggerFactory.getLogger(com.sap.cds.services.mt.impl.SubscriptionDependencyMtHandler.class);

  public static final String PORTAL = "portal";
  public static final String HTML5_APPS_REPO = "html5-apps-repo";
  private static final String XSAPPNAME_FIELD = "xsappname";
  private String xsappname;
  private String bindingName;

  public SubscriptionDependencyMtHandler(Stream<ServiceBinding> bindings, String serviceName) {
    Optional<ServiceBinding> binding =
        bindings.filter(b -> ServiceBindingUtils.matches(b, serviceName)).findFirst();
    if (binding.isPresent()) {
      this.bindingName = binding.get().getName().orElse("");

      Map<String, Object> credentials = binding.get().getCredentials();
      if (credentials.get("uaa") instanceof Map<?, ?> uaa && uaa.containsKey(XSAPPNAME_FIELD)) {
        this.xsappname = (String) uaa.get(XSAPPNAME_FIELD);
      }
    }
  }

  @After
  public void addDestinationDependency(DependenciesEventContext context) {
    if (xsappname == null) {
      return;
    }

    logger.debug(
        "Providing subscription dependency for binding '{}' with xsappname '{}'",
        this.bindingName,
        xsappname);
    context.getResult().add(Collections.singletonMap(XSAPPNAME_FIELD, xsappname));
  }
}
