/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata;

import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import com.sap.cds.integration.cloudsdk.destination.RemoteServiceHttpClientProvider;
import com.sap.cds.services.environment.CdsProperties.Remote.RemoteServiceConfig.Http;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Scope;
import org.apache.http.client.HttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.RemoteService;
import com.sap.cds.services.environment.CdsProperties.Remote.RemoteServiceConfig;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OpenTelemetryUtils;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataResponseException;

public class RemoteODataClient {

	private static final Logger logger = LoggerFactory.getLogger(RemoteODataClient.class);
	private static final String RESULT_CONTEXT_KEY = "result";

	@FunctionalInterface
	public static interface RequestExecutor {

		public Object execute(HttpClient client, Http httpConfig, ODataProtocol protocol, String servicePath);

	}

	public void processEvent(EventContext context, RequestExecutor executor, BiConsumer<Optional<Span>, String> attributesProvider) {
		String serviceName = context.getService().getName();
		RemoteServiceConfig remoteServiceConfig = context.getCdsRuntime().getEnvironment().getCdsProperties().getRemote().getService(serviceName);
		ODataProtocol protocol = getProtocol(remoteServiceConfig);
		if(protocol == null) {
			return;
		}

		Optional<Span> span = OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.CQN);

		try (Scope scope = span.map(Span::makeCurrent).orElse(null)) {
			attributesProvider.accept(span, (protocol == ODataProtocol.V2 ? RemoteServiceConfig.ODATA_V2_TYPE : RemoteServiceConfig.ODATA_V4_TYPE));

			HttpClient client = RemoteServiceHttpClientProvider.getHttpClient(remoteServiceConfig, context.getCdsRuntime());
			String servicePath = getServicePath(remoteServiceConfig.getHttp(), context);

			logger.debug("CQN >>{}<<", context.get("cqn"));
			Object result = handleExceptions(() -> {
				return executor.execute(client, remoteServiceConfig.getHttp(), protocol, servicePath);
			});
			context.put(RESULT_CONTEXT_KEY, result);
			context.setCompleted();
		} catch(Exception e) {
			OpenTelemetryUtils.recordException(span, e);
			throw e;
		} finally {
			OpenTelemetryUtils.endSpan(span);
		}
	}

	private String getServicePath(Http httpConfig, EventContext context) {
		String servicePath;
		if(StringUtils.isEmpty(httpConfig.getService())) {
			servicePath = ((RemoteService) context.getService()).getDefinition().getQualifiedName();
		} else {
			servicePath = httpConfig.getService().trim();
		}

		if(!StringUtils.isEmpty(httpConfig.getSuffix())) {
			servicePath = httpConfig.getSuffix().trim() + "/" + servicePath;
		}
		return servicePath;
	}

	private ODataProtocol getProtocol(RemoteServiceConfig remoteServiceConfig) {
		if(RemoteServiceConfig.ODATA_V2_TYPE.equalsIgnoreCase(remoteServiceConfig.getType())) {
			return ODataProtocol.V2;
		} else if (RemoteServiceConfig.ODATA_V4_TYPE.equalsIgnoreCase(remoteServiceConfig.getType())) {
			return ODataProtocol.V4;
		}
		return null;
	}

	private <T> T handleExceptions(Supplier<T> executor) {
		try {
			return executor.get();
		} catch (ODataResponseException e) {
			int statusCode = e.getHttpCode();
			String errorMessage = e.getHttpBody().getOrElse("");
			if(statusCode >= 500 && statusCode < 600) {
				logger.error("Received OData response with status code '{}' and error message '{}'", statusCode, errorMessage);
			} else {
				logger.debug("Received OData response with status code '{}' and error message '{}'", statusCode, errorMessage);
			}

			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_ERROR_RESPONSE, statusCode, e);
		}
	}

}