/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.CdsData;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.RemoteService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.odata.query.StructuredQueryBuilder;
import com.sap.cds.services.impl.odata.serialization.ODataJsonSerializer;
import com.sap.cds.services.impl.odata.uri.ETagExtractor;
import com.sap.cds.services.impl.odata.uri.UriGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.impl.odata.utils.CqnToCloudSdkConverter;
import com.sap.cds.services.impl.odata.utils.ODataDataUtils;
import com.sap.cds.services.impl.odata.utils.ODataRequestUtils;
import com.sap.cds.services.impl.odata.utils.ODataTypeUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OpenTelemetryUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.util.ProjectionResolver;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataResponseException;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions.OperandSingle;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ODataResourcePath;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataFunctionParameters;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestAction;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestBatch;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestCreate;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestDelete;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestFunction;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestRead;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestResultGeneric;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestResultMultipartGeneric;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestUpdate;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataUriFactory;
import com.sap.cloud.sdk.datamodel.odata.client.request.UpdateStrategy;

@ServiceName(value = "*", type = RemoteService.class)
public class RemoteODataHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(RemoteODataHandler.class);
	private final RemoteODataClient client = new RemoteODataClient();

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void read(CdsReadEventContext context) {
		client.processEvent(context, (httpClient, config, protocol, servicePath) -> {
			ProjectionResolver<CqnSelect> resolver = resolveProjection(context.getCqn(), context);
			CqnSelect projected = resolver.getResolvedStatement();
			CdsEntity projectedTarget = getTargetEntityFromProjection(projected, context);
			ConversionContext cc = new ConversionContext(context.getCqnNamedValues(), protocol, context);
			UriGenerator uriGenerator = new UriGenerator(cc);
			ODataResourcePath entityPath = uriGenerator.analyze(projected);
			boolean isCollection = uriGenerator.isCollection();

			StructuredQueryBuilder sqb = StructuredQueryBuilder.forEntity(cc, entityPath.toEncodedPathString(), projectedTarget);
			handleQueryParameters(projected, isCollection, sqb);
			StructuredQuery query = sqb.build();

			ODataRequestUtils.prepareQuery(query, context);

			logger.debug("GET >>{}{}?{}<<", servicePath, entityPath, query.getQueryString());

			ODataRequestResultGeneric result;
			try {
				ODataRequestRead readRequest = new ODataRequestRead(servicePath, entityPath, query.getEncodedQueryString(), protocol);
				ODataRequestUtils.prepareRequest(readRequest, config, context);
				result = readRequest.execute(httpClient);
			} catch (ODataResponseException e) {
				if(!uriGenerator.isCollection() && isEmptyResult(e.getHttpCode())) {
					return ResultBuilder.selectedRows(Collections.emptyList()).result();
				}
				throw e;
			}

			List<CdsData> resultList = isCollection
					? ODataDataUtils.entityCollection(result, projectedTarget, protocol)
					: Arrays.asList(ODataDataUtils.entity(result, projectedTarget, protocol));
			ResultBuilder resultBuilder = ResultBuilder.selectedRows(resolver.transform(resultList));
			if (isCollection && projected.hasInlineCount()) {
				resultBuilder.inlineCount(result.getInlineCount());
			}
			return ODataTypeUtils.toCdsTypes(context.getTarget(), resultBuilder.result());
		}, (span, protocol) -> OpenTelemetryUtils.updateSpan(span, context.getCdsRuntime(), "SELECT", context.getTarget(), context.getCqn(), protocol));
	}

	@VisibleForTesting
	void handleQueryParameters(CqnSelect projected, boolean isCollection, StructuredQueryBuilder sqb) {
		sqb.select(projected);
		sqb.expand(projected);

		// these query parameters are only supported for collections
		if (isCollection) {
			sqb.filter(projected.where());
			sqb.search(projected.search());
			sqb.orderBy(projected.orderBy());
			sqb.top(projected);
			sqb.skip(projected);
			sqb.inlineCount(projected);
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void create(CdsCreateEventContext context) {
		client.processEvent(context, (httpClient, config, protocol, servicePath) -> {
			ODataDataUtils.clean(context.getCqn().entries(), context.getTarget());

			ProjectionResolver<CqnInsert> resolver = resolveProjection(context.getCqn(), context);
			CqnInsert projected = resolver.getResolvedStatement();
			CdsEntity projectedTarget = getTargetEntityFromProjection(projected, context);
			ConversionContext cc = new ConversionContext(protocol, context);
			UriGenerator uriGenerator = new UriGenerator(cc);
			ODataResourcePath entityPath = uriGenerator.analyze(projected);
			logger.debug("POST >>{}{}<<", servicePath, entityPath);

			if(!uriGenerator.isCollection()) {
				throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_ONLY_COLLECTION);
			}

			List<Map<String, Object>> result = new ArrayList<>();
			ODataJsonSerializer serializer = ODataJsonSerializer.create(protocol);

			List<ODataRequestCreate> requests = new ArrayList<>();
			for(Map<String, Object> entry : projected.entries()) {
				String serializedEntry = serializer.serialize(entry);
				ODataRequestCreate request = new ODataRequestCreate(servicePath, entityPath, serializedEntry, protocol);
				ODataRequestUtils.prepareRequest(request, config, context);
				requests.add(request);
			}

			if(requests.size() == 1) {
				ODataRequestResultGeneric requestResult = requests.get(0).execute(httpClient);
				result.add(ODataDataUtils.entity(requestResult, projectedTarget, protocol));

			} else if (requests.size() > 1) {
				ODataRequestBatch.Changeset changeset = new ODataRequestBatch(servicePath, protocol).beginChangeset();
				requests.forEach(changeset::addCreate);

				ODataRequestBatch batchRequest = changeset.endChangeset();
				ODataRequestUtils.prepareRequest(batchRequest, config, context);

				ODataRequestResultMultipartGeneric batchResult = batchRequest.execute(httpClient);
				for(ODataRequestCreate request : requests) {
					result.add(ODataDataUtils.entity(batchResult.getResult(request), projectedTarget, protocol));
				}
			}
			Result transformedResult = ResultBuilder.insertedRows(resolver.transform(result)).result();
			return ODataTypeUtils.toCdsTypes(context.getTarget(), transformedResult);
		}, (span, protocol) -> OpenTelemetryUtils.updateSpan(span, context.getCdsRuntime(), "INSERT", context.getTarget(), context.getCqn(), protocol));
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void update(CdsUpdateEventContext context) {
		client.processEvent(context, (httpClient, config, protocol, servicePath) -> {
			ODataDataUtils.clean(context.getCqn().entries(), context.getTarget());

			ProjectionResolver<CqnUpdate> resolver = resolveProjection(context.getCqn(), context);
			CqnUpdate projected = resolver.getResolvedStatement();
			CdsEntity projectedTarget = getTargetEntityFromProjection(projected, context);
			ODataJsonSerializer serializer = ODataJsonSerializer.create(protocol);
			int dataBatches = projected.entries().size();
			List<Map<String, Object>> batches = valueSetsOrEmptyParameters(context.getCqnValueSets(), dataBatches);
			if(dataBatches != 1 && dataBatches != batches.size()) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_BATCH_UPDATE, batches.size(), dataBatches);
			}

			int[] updated = new int[batches.size()];
			List<Map<String, Object>> updatedData = new ArrayList<>(batches.size());

			List<ODataRequestUpdate> requests = new ArrayList<>(batches.size());
			List<Map<String, Object>> requestEntries = new ArrayList<>(batches.size());
			for(int i=0; i<batches.size(); ++i) {
				ConversionContext cc = new ConversionContext(batches.get(i), protocol, context);
				UriGenerator uriGenerator = new UriGenerator(cc);

				Map<String, Object> entry = projected.entries().get(dataBatches > 1 ? i : 0);
				ODataResourcePath entityPath = uriGenerator.analyze(projected, entry);
				logger.debug("PATCH >>{}{}<<", servicePath, entityPath);

				if(uriGenerator.isCollection()) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_ONLY_ENTITY);
				}

				String serializedEntry = serializer.serialize(entry);
				ODataRequestUpdate request = new ODataRequestUpdate(servicePath, entityPath, serializedEntry, UpdateStrategy.MODIFY_WITH_PATCH, null, protocol);
				ODataRequestUtils.prepareRequest(request, config, context);
				ETagExtractor.create(cc).setIfMatch(request, projected.where(), entry);
				requests.add(request);
				requestEntries.add(entry);
			}

			if(requests.size() == 1) {
				try {
					ODataRequestResultGeneric requestResult = requests.get(0).execute(httpClient);
					updated[0] = 1;

					boolean noContent = requestResult.getHttpResponse().getStatusLine().getStatusCode() == HttpStatus.SC_NO_CONTENT;
					updatedData.add(noContent ? ODataDataUtils.noContent(requestResult, requestEntries.get(0)) : ODataDataUtils.entity(requestResult, projectedTarget, protocol));
				} catch (ODataResponseException e) {
					if (isEmptyResult(e.getHttpCode())) {
						updated[0] = 0;
						updatedData.add(new HashMap<>());
					} else {
						throw e;
					}
				}

			} else if (requests.size() > 1) {
				ODataRequestBatch.Changeset changeset = new ODataRequestBatch(servicePath, protocol).beginChangeset();
				requests.forEach(changeset::addUpdate);

				ODataRequestBatch batchRequest = changeset.endChangeset();
				ODataRequestUtils.prepareRequest(batchRequest, config, context);

				ODataRequestResultMultipartGeneric batchResult = batchRequest.execute(httpClient);
				for(int i=0; i<requests.size(); ++i) {
					try {
						ODataRequestResultGeneric requestResult = batchResult.getResult(requests.get(i));
						updated[i] = 1;

						boolean noContent = requestResult.getHttpResponse().getStatusLine().getStatusCode() == HttpStatus.SC_NO_CONTENT;
						updatedData.add(noContent ? ODataDataUtils.noContent(requestResult, requestEntries.get(i)) : ODataDataUtils.entity(requestResult, projectedTarget, protocol));
					} catch (ODataResponseException e) {
						if (isEmptyResult(e.getHttpCode())) {
							updated[i] = 0;
							updatedData.add(new HashMap<>());
							continue;
						}
						throw e;
					}
				}
			}
			ResultBuilder transformedResultBuilder = ResultBuilder.batchUpdate();
			List<? extends Map<String, Object>> transformedData = resolver.transform(updatedData);
			for (int i = 0; i < updated.length; ++i) {
				transformedResultBuilder.addUpdatedRows(updated[i], transformedData.get(i));
			}
			Result transformedResult = transformedResultBuilder.result();
			return ODataTypeUtils.toCdsTypes(context.getTarget(), transformedResult);
		}, (span, protocol) -> OpenTelemetryUtils.updateSpan(span, context.getCdsRuntime(), "UPDATE", context.getTarget(), context.getCqn(), protocol));
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void delete(CdsDeleteEventContext context) {
		client.processEvent(context, (httpClient, config, protocol, servicePath) -> {
			CqnDelete projected = resolveProjection(context.getCqn(), context).getResolvedStatement();
			List<Map<String, Object>> batches = valueSetsOrEmptyParameters(context.getCqnValueSets(), 1);
			int[] deleted = new int[batches.size()];

			List<ODataRequestDelete> requests = new ArrayList<>(batches.size());
			for(int i=0; i<batches.size(); ++i) {
				ConversionContext cc = new ConversionContext(batches.get(i), protocol, context);
				UriGenerator uriGenerator = new UriGenerator(cc);
				ODataResourcePath entityPath = uriGenerator.analyze(projected);
				logger.debug("DELETE >>{}{}<<", servicePath, entityPath);

				if(uriGenerator.isCollection()) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_ONLY_ENTITY);
				}

				ODataRequestDelete request = new ODataRequestDelete(servicePath, entityPath, null, protocol);
				ODataRequestUtils.prepareRequest(request, config, context);
				ETagExtractor.create(cc).setIfMatch(request, projected.where());
				requests.add(request);
			}

			if(requests.size() == 1) {
				try {
					requests.get(0).execute(httpClient);
					deleted[0] = 1;
				} catch (ODataResponseException e) {
					if (isEmptyResult(e.getHttpCode())) {
						deleted[0] = 0;
					} else {
						throw e;
					}
				}

			} else if (requests.size() > 1) {
				ODataRequestBatch.Changeset changeset = new ODataRequestBatch(servicePath, protocol).beginChangeset();
				requests.forEach(changeset::addDelete);

				ODataRequestBatch batchRequest = changeset.endChangeset();
				ODataRequestUtils.prepareRequest(batchRequest, config, context);

				ODataRequestResultMultipartGeneric batchResult = batchRequest.execute(httpClient);
				for(int i=0; i<requests.size(); ++i) {
					try {
						batchResult.getResult(requests.get(i));
						deleted[i] = 1;
					} catch (ODataResponseException e) {
						if (isEmptyResult(e.getHttpCode())) {
							deleted[i] = 0;
							continue;
						}
						throw e;
					}
				}
			}

			return ResultBuilder.deletedRows(deleted).result();
		}, (span, protocol) -> OpenTelemetryUtils.updateSpan(span, context.getCdsRuntime(), "DELETE", context.getTarget(), context.getCqn(), protocol));
	}

	@On(event = "*")
	@HandlerOrder(OrderConstants.On.FEATURE + 1)
	public void handleActionAndFunction(EventContext context) {
		String event = context.getEvent();
		CdsService service = ((RemoteService) context.getService()).getDefinition();

		Optional<CdsAction> action = CdsModelUtils.findAction(service, context.getTarget(), context.getEvent());
		Optional<CdsFunction> function = CdsModelUtils.findFunction(service, context.getTarget(), context.getEvent());
		if (!(action.isPresent() || function.isPresent())) {
			logger.debug("Found no function or action matching '{}' in service '{}' for target '{}'", event, service, context.getTarget());
			return;
		}

		client.processEvent(context, (httpClient, config, protocol, servicePath) -> {
			CqnSelect select = (CqnSelect) context.get("cqn");
			CdsEntity target = context.getTarget();

			String operation = action.map(a -> a.getQualifiedName()).orElse(function.map(f -> f.getQualifiedName()).orElse(null));
			boolean isServiceBound = operation.startsWith(service.getQualifiedName());
			if (!isServiceBound && select == null) {
				throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_MISSING_STATEMENT);
			}

			String operationName;
			if (isServiceBound) {
				operationName = context.getEvent();
			} else if (protocol == ODataProtocol.V4) {
				operationName = service.getQualifiedName() + "." + context.getEvent();
			} else {
				operationName = target.getName() + "_" + operation;
			}

			ODataResourcePath resourcePath;
			ConversionContext cc = new ConversionContext(protocol, context);
			if (protocol == ODataProtocol.V2) {
				resourcePath = ODataResourcePath.of(operationName);
			} else {
				UriGenerator uriGenerator = new UriGenerator(cc, true);
				resourcePath = uriGenerator.analyze(select);
				resourcePath.addSegment(operationName);
			}

			CdsType type;
			ODataRequestResultGeneric result;
			if (action.isPresent()) {
				type = action.get().returnType().orElse(null);
				ODataRequestAction request;
				if (protocol == ODataProtocol.V4) {
					ODataJsonSerializer serializer = ODataJsonSerializer.create(protocol);
					Map<String, Object> parameters = new HashMap<>();
					action.get().parameters().forEach(e -> parameters.put(e.getName(), context.get(e.getName())));
					String actionParameters = serializer.serialize(parameters);
					request = new ODataRequestAction(servicePath, resourcePath, actionParameters, protocol);
					logger.debug("POST >>{}{}<<", servicePath, resourcePath);
				} else {
					// In V2 there are no actual actions, therefore we basically send a function request with POST
					Map<String, OperandSingle> parameters = action.get().parameters().filter(param -> context.get(param.getName()) != null)
							.collect(Collectors.toMap(CdsParameter::getName
									, e -> CqnToCloudSdkConverter.convert(context.get(e.getName()), null, e.getType().as(CdsSimpleType.class).getType(), cc)));
					request = new ODataRequestAction(servicePath, resourcePath, null, protocol);
					if (select != null) {
						// add keys to the parameters
						CqnAnalyzer analyzer = CqnAnalyzer.create(context.getModel());
						AnalysisResult analysisResult = analyzer.analyze(select);
						Map<String, Object> keys = analysisResult.targetKeys();
						target.keyElements().forEach(key -> {
							parameters.put(key.getName(), CqnToCloudSdkConverter.convert(keys.get(key.getName()), target, key.getType().as(CdsSimpleType.class).getType(), cc));
						});
					}
					parameters.entrySet().stream().forEach(entry -> {
						request.addQueryParameter(entry.getKey()
								, ODataUriFactory.encodeQuery(entry.getValue().getExpression(protocol)));
					});
					logger.debug("POST >>{}{}?{}<<", servicePath, resourcePath, request.getRequestQuery());
				}
				ODataRequestUtils.prepareRequest(request, config, context);
				result = request.execute(httpClient);
			} else {
				type = function.get().getReturnType();
				Map<String, Object> parameters = function.get().parameters()
						.filter(param -> context.get(param.getName()) != null || protocol == ODataProtocol.V4)
						.collect(Collectors.toMap(CdsParameter::getName
								, e -> CqnToCloudSdkConverter.convert(context.get(e.getName()), null, e.getType().as(CdsSimpleType.class).getType(), cc)));
				if (protocol == ODataProtocol.V2 && select != null) {
					// add keys to the parameters
					CqnAnalyzer analyzer = CqnAnalyzer.create(context.getModel());
					AnalysisResult analysisResult = analyzer.analyze(select);
					Map<String, Object> keys = analysisResult.targetKeys();
					target.keyElements().forEach(key -> {
						parameters.put(key.getName(), CqnToCloudSdkConverter.convert(keys.get(key.getName()), target, key.getType().as(CdsSimpleType.class).getType(), cc));
					});
				}
				ODataFunctionParameters odataParameters = ODataFunctionParameters.of(parameters, protocol);
				ODataRequestFunction request = new ODataRequestFunction(servicePath, resourcePath, odataParameters, null, protocol);
				logger.debug("GET >>{}{}?{}<<", servicePath, resourcePath, request.getQuery());
				ODataRequestUtils.prepareRequest(request, config, context);
				result = request.execute(httpClient);
			}
			if (result.getHttpResponse().getStatusLine().getStatusCode() == 204) {
				return null;
			}

			return ODataDataUtils.operation(result, type, protocol, operationName);
		}, (span, protocol) -> { });
	}

	private <T extends CqnStatement> ProjectionResolver<T> resolveProjection(T cqn, EventContext context) {
		CqnAnalyzer analyzer = CqnAnalyzer.create(context.getModel());
		String serviceName = ((RemoteService) context.getService()).getDefinition().getQualifiedName();

		ProjectionResolver<T> resolver = ProjectionResolver.create(context.getModel(), cqn)
				.condition((prev, curr, next) -> {
					// stop if next projection doesn't match anymore
					return !analyzer.analyze(next.ref()).rootEntity().getQualifier().equals(serviceName);
				})
				.resolveAliases() // force alias resolvement
				.resolveAll();
		logger.debug("CQN (projected) >>{}<<", resolver.getResolvedStatement());
		return resolver;
	}

	private CdsEntity getTargetEntityFromProjection(CqnStatement cqn, EventContext context) {
		return CdsModelUtils.getEntityPath(cqn.ref(), context.getModel()).target().entity();
	}

	private List<Map<String, Object>> valueSetsOrEmptyParameters(Iterable<Map<String, Object>> cqnValueSets, int defaultBatchSize) {
		List<Map<String, Object>> valueSetsOrEmptyParameters = new ArrayList<>();
		cqnValueSets.forEach(valueSetsOrEmptyParameters::add);
		if(valueSetsOrEmptyParameters.isEmpty()) {
			for(int i=0; i<defaultBatchSize; ++i) {
				valueSetsOrEmptyParameters.add(Collections.emptyMap());
			}
		}
		return valueSetsOrEmptyParameters;
	}

	private boolean isEmptyResult(int statusCode) {
		boolean isEmpty = statusCode == HttpStatus.SC_NOT_FOUND || statusCode == HttpStatus.SC_PRECONDITION_FAILED;
		if (isEmpty) {
			logger.debug("Received OData response with status code '{}': Returning empty result", statusCode);
		}
		return isEmpty;
	}

}
