/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.query;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.impl.odata.utils.AbstractGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;

/**
 * An {@link AbstractGenerator} that converts CQN expands to an OData query
 * option
 */
public class ExpandGenerator implements AbstractGenerator, CqnVisitor {

	private final List<StructuredQuery> expanded = new ArrayList<>();
	private final Set<String> excluded = new HashSet<>();
	private final CdsEntity entity;
	private final ConversionContext context;

	public ExpandGenerator(CdsEntity entity, ConversionContext context) {
		this.entity = entity;
		this.context = context;
	}

	public ExpandGenerator(CdsEntity entity, ConversionContext context, CqnSelect select) {
		this(entity, context);
		this.excluded.addAll(select.excluding());
	}

	@Override
	public void visit(CqnExpand expand) {
		CqnStructuredTypeRef ref = expand.ref();
		if (ref.size() != 1) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_PATH_EXPR);
		}
		ODataProtocol protocol = context.getProtocol();
		CdsEntity targetEntity = entity.getTargetOf(ref.firstSegment());
		StructuredQueryBuilder sqb = StructuredQueryBuilder.forSubquery(context, ref.firstSegment(), targetEntity);

		sqb.expand(expand.items());
		sqb.select(expand.items());

		// these query parameters are only supported for collections
		if (protocol == ODataProtocol.V4 && !CdsModelUtils.isSingleValued(entity.getAssociation(ref.firstSegment()).getType())) {
			sqb.filter(expand.ref().rootSegment().filter());
			sqb.orderBy(expand.orderBy());
			sqb.top(expand);
			sqb.skip(expand);
		}

		expanded.add(sqb.build());
	}

	@Override
	public void visit(CqnSelect select) {
		// must not be called
		throw new IllegalStateException();
	}

	@Override
	public void apply(StructuredQuery query) {
		expanded.removeIf(q -> excluded.contains(q.getEntityOrPropertyName()));
		if (expanded.size() > 0) {
			query.select(expanded.toArray(new StructuredQuery[expanded.size()]));
		}
	}

}
