/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.query;

import java.util.List;

import com.sap.cds.impl.util.Stack;
import com.sap.cds.ql.BooleanFunction;
import com.sap.cds.ql.Value;
import com.sap.cds.ql.cqn.CqnBetweenPredicate;
import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnContainmentTest;
import com.sap.cds.ql.cqn.CqnFunc;
import com.sap.cds.ql.cqn.CqnInPredicate;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnNegation;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.impl.odata.utils.AbstractGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.impl.odata.utils.CqnToCloudSdkConverter;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.util.CqnStatementUtils;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions.Operand;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions.OperandSingle;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FilterableComparisonAbsolute;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FilterableComparisonRelative;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueBoolean;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueCollection;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;

public class FilterGenerator implements AbstractGenerator, CqnVisitor {

	private final CdsEntity entity;
	private final ConversionContext context;

	/**
	 * Map that stores the conversion results of the individual predicates
	 */
	private final Stack<ValueBoolean> expression = new Stack<>();

	public FilterGenerator(CdsEntity entity, ConversionContext context) {
		this.entity = entity;
		this.context = context;
	}

	@Override
	public void visit(CqnComparisonPredicate comparison) {
		if (comparison.left().isList() && comparison.right().isList()) {
			CqnStatementUtils.unfold(comparison).accept(this);
			return;
		}

		CqnValue left = comparison.left();
		CqnValue right = comparison.right();

		CqnToCloudSdkConverter converter = new CqnToCloudSdkConverter(left, right, entity, context);
		converter.convert();
		OperandSingle convLeft = converter.getLeft();
		OperandSingle convRight = converter.getRight();

		try {
			switch (comparison.operator()) {
			case EQ:
			case IS:
				expression.push(((FilterableComparisonAbsolute) convLeft).equalTo(convRight));
				break;
			case NE:
			case IS_NOT:
				expression.push(((FilterableComparisonAbsolute) convLeft).notEqualTo(convRight));
				break;
			case GE:
				expression.push(((FilterableComparisonRelative) convLeft).greaterThanEqual(convRight));
				break;
			case GT:
				expression.push(((FilterableComparisonRelative) convLeft).greaterThan(convRight));
				break;
			case LE:
				expression.push(((FilterableComparisonRelative) convLeft).lessThanEqual(convRight));
				break;
			case LT:
				expression.push(((FilterableComparisonRelative) convLeft).lessThan(convRight));
				break;
			}
		} catch (ClassCastException e) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_OPERATOR, comparison.operator(), e);
		}
	}

	@Override
	public void visit(CqnBetweenPredicate b) {
		try {
			CqnValue v = b.value();
			FilterableComparisonRelative btwn = (FilterableComparisonRelative) CqnToCloudSdkConverter
					.convert(v, entity, context);

			Operand lo = CqnToCloudSdkConverter.convert(b.low(), entity, v, context);
			Operand hi = CqnToCloudSdkConverter.convert(b.high(), entity, v, context);
			expression.push(btwn.greaterThanEqual(lo).and(btwn.lessThanEqual(hi)));
		} catch (ClassCastException e) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_OPERATOR, "between", e);
		}
	}

	@Override
	public void visit(CqnConnectivePredicate connective) {
		List<ValueBoolean> expressions = expression.pop(connective.predicates().size());
		ValueBoolean result = expressions.get(0);
		for(int i=1; i<expressions.size(); ++i) {
			switch (connective.operator()) {
			case AND:
				result = result.and(expressions.get(i));
				break;
			case OR:
				result = result.or(expressions.get(i));
				break;
			}
		}
		expression.push(result);
	}

	@Override
	public void visit(CqnNegation neg) {
		expression.push(expression.pop().not());
	}

	@SuppressWarnings("unchecked")
	@Override
	public void visit(CqnContainmentTest test) {
		if (test.caseInsensitive()) {
			Value<String> val = (Value<String>) test.value();
			Value<String> term = (Value<String>) test.term();
			test = (CqnContainmentTest) val.toUpper().contains(term.toUpper());
		}
		expression.push((ValueBoolean) CqnToCloudSdkConverter.convert(test, entity, context));
	}

	@Override
	public void visit(CqnInPredicate in) {
		try {
			// OData V2 doesn't support IN
			// tuple IN not supported by OData at all
			// needs to be translated into (... AND ...) OR (... AND ...) OR ...
			if(context.getProtocol() == ODataProtocol.V2 || in.value().isList()) {
				CqnStatementUtils.unfold(in).accept(this);
			} else {
				FilterableComparisonAbsolute comparable = (FilterableComparisonAbsolute) CqnToCloudSdkConverter.convert(in.value(), entity, context);
				Operand[] operands = in.values().stream().map(v -> CqnToCloudSdkConverter.convert(v, entity, in.value(), context)).toArray(Operand[]::new);
				ValueBoolean predicate = comparable.in(operands);
				expression.push(predicate);
			}

		} catch (ClassCastException e) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_OPERATOR, "in", e);
		}
	}

	@Override
	public void visit(CqnFunc func) {
		if (func instanceof BooleanFunction) {
			expression.push((ValueBoolean) CqnToCloudSdkConverter.convert(func, entity, context));
		}
	}

	@Override
	public void apply(StructuredQuery query) {
		if(!expression.isEmpty()) {
			query.filter(expression.pop());
		}
	}

	@Override
	public void visit(CqnMatchPredicate query) {
		if (context.getProtocol() == ODataProtocol.V2) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_V2_LAMBDA);
		}
		ValueCollection collection = CqnToCloudSdkConverter.convertRefToFieldUntyped(query.ref()).asCollection();
		if (query.predicate().isPresent()) {
			FilterGenerator generator = new FilterGenerator(CdsModelUtils.getRefTarget(query.ref(), entity), context);
			query.predicate().get().accept(generator);
			ValueBoolean odataQuery = generator.expression.pop();
			switch (query.quantifier()) {
			case ANY:
				expression.push(collection.any(odataQuery));
				break;
			case ALL:
				expression.push(collection.all(odataQuery));
			}
		} else {
			switch (query.quantifier()) {
			case ANY:
				expression.push(collection.any());
				break;
			case ALL:
				throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_INVALID_LAMBDA_ALL);
			}
		}
	}

	@Override
	public void visit(CqnSelect select) {
		// must not be called
		throw new IllegalStateException();
	}

}
