/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.query;

import java.util.Locale;

import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.impl.odata.utils.AbstractGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.impl.odata.utils.CqnToCloudSdkConverter;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions.OperandSingle;
import com.sap.cloud.sdk.datamodel.odata.client.expression.OrderExpression;
import com.sap.cloud.sdk.datamodel.odata.client.query.Order;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;

/**
 * An {@link AbstractGenerator} that handles the $orderby clause.
 */
public class OrderByGenerator implements AbstractGenerator, CqnVisitor {

	private OrderExpression orderExpr = null;
	private final CdsEntity entity;
	private final ConversionContext context;

	public OrderByGenerator(CdsEntity entity, ConversionContext context) {
		this.entity = entity;
		this.context = context;
	}

	@Override
	public void visit(CqnSortSpecification sortSpec) {
		OperandSingle operand = CqnToCloudSdkConverter.convert(sortSpec.value(), entity, context);
		String item = operand.getExpression(context.getProtocol());
		if (sortSpec.order() != null) {
			addOrderExpr(item, Order.valueOf(sortSpec.order().toString().toUpperCase(Locale.US)));
		} else {
			addOrderExpr(item, Order.ASC);
		}
	}

	private void addOrderExpr(String fieldName, Order order) {
		if (orderExpr == null) {
			orderExpr = OrderExpression.of(fieldName, order);
		} else {
			orderExpr = orderExpr.and(fieldName, order);
		}
	}

	@Override
	public void apply(StructuredQuery query) {
		if (orderExpr != null) {
			query.orderBy(orderExpr);
		}
	}

}
