/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.query;

import java.util.List;
import java.util.Locale;

import com.sap.cds.impl.util.Stack;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnNegation;
import com.sap.cds.ql.cqn.CqnSearchTermPredicate;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.services.impl.odata.utils.AbstractGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;

public class SearchGenerator implements AbstractGenerator, CqnVisitor {

	private final ConversionContext context;
	private final Stack<String> search = new Stack<>();

	public SearchGenerator(ConversionContext context) {
		this.context = context;
	}

	@Override
	public void visit(CqnSearchTermPredicate searchPredicate) {
		String escapedTerm = escapeDoubleQuotes(searchPredicate.searchTerm());
		search.push("\"" + escapedTerm + "\"");
	}

	// TODO visit CqnPassThroughSearchPredicate

	@Override
	public void visit(CqnConnectivePredicate connective) {
		List<String> expressions = search.pop(connective.predicates().size());
		String operator = connective.operator().toString().toUpperCase(Locale.ENGLISH);
		search.push("(" + String.join(" " + operator + " ", expressions) + ")");
	}

	@Override
	public void visit(CqnNegation neg) {
		search.push("NOT(" + search.pop() + ")");
	}

	@Override
	public void apply(StructuredQuery query) {
		if (!search.isEmpty()) {
			// in V2 we use the SAP proprietary search, which has the same grammar as
			// $search
			String searchExpression = search.pop();
			if (context.getProtocol() == ODataProtocol.V4) {
				query.search(searchExpression);
			} else {
				query.withCustomParameter("search", searchExpression);
			}
		}
	}

	/**
	 * Escapes the escape characters in the given string
	 *
	 * @param s the string to escape
	 * @return the string with the escaped escape characters
	 */
	private static String escapeEscapeCharacter(String s) {
		String escaped = "\\\\";
		return s.replace("\\", escaped).replace("%5C", escaped);
	}

	/**
	 * Escapes the double quote characters in the given string
	 *
	 * @param s the string to escape
	 * @return the string with the escaped double quote characters
	 */
	private static String escapeDoubleQuotes(String s) {
		String escaped = "\\\"";
		return escapeEscapeCharacter(s).replace("\"", escaped).replace("%22", escaped);
	}

}
