/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.query;

import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnStar;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsAnnotation;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.impl.odata.utils.AbstractGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CqnStatementUtils;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;

public class SelectGenerator implements AbstractGenerator, CqnVisitor {

	private final CdsEntity target;
	private final ConversionContext context;
	private final Set<String> selected = new LinkedHashSet<>();
	private final Set<String> expanded = new LinkedHashSet<>();
	private final Set<String> excluded = new HashSet<>();
	private boolean hasStar = false;

	public SelectGenerator(CdsEntity target, ConversionContext context) {
		this.target = target;
		this.context = context;
	}

	public SelectGenerator(CdsEntity target, ConversionContext context, CqnSelect select) {
		this(target, context);
		excluded.addAll(select.excluding());
	}

	@Override
	public void visit(CqnStar star) {
		hasStar = true;
	}

	@Override
	public void visit(CqnSelectListValue slv) {
		if(slv.value().isRef()) {
			CqnElementRef ref = slv.value().asRef();
			if(ref.size() == 1) {
				String element = ref.firstSegment();
				selected.add(element);
			} else {
				throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_PATH_EXPR);
			}
		}
	}

	@Override
	public void visit(CqnExpand expand) {
		CqnStructuredTypeRef ref = expand.ref();
		if (ref.size() != 1) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_PATH_EXPR);
		}
		if (context.getProtocol() == ODataProtocol.V2 && CqnStatementUtils.isSelectStar(expand.items())) {
			selected.add(ref.firstSegment());
		} else if (context.getProtocol() == ODataProtocol.V4) {
			expanded.add(ref.firstSegment());
		}
	}

	@Override
	public void apply(StructuredQuery query) {
		if(!excluded.isEmpty()) {
			if(hasStar || selected.isEmpty()) {
				target.elements().filter(
					// already filter foreign keys for excluded elements
					e -> e.findAnnotation("odata.foreignKey4")
					.map(CdsAnnotation::getValue)
					.map(v -> !excluded.contains(v))
					.orElse(true)
				).map(e -> e.getName()).forEach(selected::add);
			}
			selected.removeAll(excluded);
			if(selected.isEmpty()) {
				throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_EMPTY_SELECTION);
			}
		} else if (context.getProtocol() == ODataProtocol.V2 && hasStar && query.isRoot()) {
			// explicit * to avoid expands to override selection of all simple elements
			selected.add("*");
		} else if (context.getProtocol() == ODataProtocol.V4 && !hasStar && selected.isEmpty() && !expanded.isEmpty()) {
			// if only expands are selected, list them explicitly in $select
			// this ensures that no simple properties are selected
			selected.addAll(expanded);
		}

		if (selected.size() > 0) {
			query.select(selected.toArray(new String[selected.size()]));
		}
	}

}
