/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.query;

import java.util.List;
import java.util.Optional;

import com.sap.cds.ql.cqn.CqnEntitySelector;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnToken;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.impl.odata.utils.AbstractGenerator;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;

public class StructuredQueryBuilder {

	private final ConversionContext context;
	private final CdsEntity target;
	private final StructuredQuery query;

	private StructuredQueryBuilder(ConversionContext context, CdsEntity targetEntity, StructuredQuery query) {
		this.context = context;
		this.target = targetEntity;
		this.query = query;
	}

	public static StructuredQueryBuilder forEntity(ConversionContext context, String entityName,
			CdsEntity targetEntity) {
		StructuredQuery q = StructuredQuery.onEntity(entityName, context.getProtocol());

		return new StructuredQueryBuilder(context, targetEntity, q);
	}

	public static StructuredQueryBuilder forSubquery(ConversionContext context, String fieldName,
			CdsEntity targetEntity) {
		StructuredQuery q = StructuredQuery.asNestedQueryOnProperty(fieldName, context.getProtocol());

		return new StructuredQueryBuilder(context, targetEntity, q);
	}

	public void skip(CqnEntitySelector selector) {
		apply(new SkipGenerator(selector));
	}

	public void top(CqnEntitySelector selector) {
		apply(new TopGenerator(selector));
	}

	public void orderBy(List<CqnSortSpecification> orderBy) {
		applyAll(orderBy, new OrderByGenerator(target, context));
	}

	public void filter(Optional<CqnPredicate> where) {
		where.ifPresent(w -> apply(w, new FilterGenerator(target, context)));
	}

	public void search(Optional<CqnPredicate> search) {
		search.ifPresent(w -> apply(w, new SearchGenerator(context)));
	}

	public void inlineCount(CqnSelect select) { // TODO -> entity selector
		apply(new InlineCountGenerator(context, select));
	}

	public void select(List<CqnSelectListItem> slis) {
		applyAll(slis, new SelectGenerator(target, context));
	}

	public void select(CqnSelect select) {
		SelectGenerator gen = new SelectGenerator(target, context, select);
		applyAll(select.items(), gen);
	}

	public void expand(List<CqnSelectListItem> slis) {
		applyAll(slis, new ExpandGenerator(target, context));
	}

	public void expand(CqnSelect select) {
		ExpandGenerator gen = new ExpandGenerator(target, context, select);
		applyAll(select.items(), gen);
	}

	public StructuredQuery build() {
		return query;
	}

	private void apply(AbstractGenerator gen) {
		gen.apply(query);
	}

	private <G extends AbstractGenerator & CqnVisitor> void apply(CqnToken t, G gen) {
		t.accept(gen);
		apply(gen);
	}

	private <G extends AbstractGenerator & CqnVisitor> void applyAll(Iterable<? extends CqnToken> tokens, G gen) {
		tokens.forEach(t -> t.accept(gen));
		apply(gen);
	}

}
