/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.serialization;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

class ODataV4JsonSerializer implements ODataJsonSerializer {

	private static ObjectMapper mapper = new ObjectMapper()
			.setSerializationInclusion(Include.ALWAYS)
			.registerModule(new SimpleModule()
					.addSerializer(new InstantSerializer(Instant.class))
					.addSerializer(new LocalDateSerializer(LocalDate.class))
					.addSerializer(new LocalTimeSerializer(LocalTime.class)));

	@Override
	public String serialize(Map<String, Object> row) {
		try {
			return mapper.writeValueAsString(row);
		} catch (JsonProcessingException e) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_JSON_SERIALIZATION, e);
		}
	}

	private static class InstantSerializer extends StdSerializer<Instant> {

		private static final long serialVersionUID = 1L;

		public InstantSerializer(Class<Instant> t) {
			super(t);
		}

		@Override
		public void serialize(Instant instant, JsonGenerator gen, SerializerProvider provider) throws IOException {
			gen.writeString(DateTimeFormatter.ISO_INSTANT.format(instant));
		}
	}

	private static class LocalDateSerializer extends StdSerializer<LocalDate> {

		private static final long serialVersionUID = 1L;

		public LocalDateSerializer(Class<LocalDate> t) {
			super(t);
		}

		@Override
		public void serialize(LocalDate localDate, JsonGenerator gen, SerializerProvider provider) throws IOException {
			gen.writeString(DateTimeFormatter.ISO_DATE.format(localDate));
		}
	}

	private static class LocalTimeSerializer extends StdSerializer<LocalTime> {

		private static final long serialVersionUID = 1L;

		public LocalTimeSerializer(Class<LocalTime> t) {
			super(t);
		}

		@Override
		public void serialize(LocalTime localTime, JsonGenerator gen, SerializerProvider provider) throws IOException {
			gen.writeString(DateTimeFormatter.ISO_TIME.format(localTime));
		}
	}

}
