/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.uri;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.sap.cds.CdsData;
import com.sap.cds.ql.cqn.CqnEtagPredicate;
import com.sap.cds.ql.cqn.CqnListValue;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.util.OccUtils;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestGeneric;

public class ETagExtractor {

	public static final String METADATA_ETAG_KEY = "etag";

	private final ConversionContext context;

	private ETagExtractor(ConversionContext context) {
		this.context = context;
	}

	public static ETagExtractor create(ConversionContext context) {
		return new ETagExtractor(context);
	}

	public void setIfMatch(ODataRequestGeneric request, Optional<CqnPredicate> predicate) {
		setIfMatch(request, predicate, null);
	}

	public void setIfMatch(ODataRequestGeneric request, Optional<CqnPredicate> predicate, Map<String, Object> entry) {
		List<String> values = extractETagValues(predicate, entry);
		if (!values.isEmpty()) {
			request.setHeader(HttpHeaders.IF_MATCH, values);
		}
	}

	private List<String> extractETagValues(Optional<CqnPredicate> predicate, Map<String, Object> entry) {
		Optional<CqnEtagPredicate> optETag = predicate.flatMap(OccUtils::eTagPredicate);
		if (optETag.isPresent()) {
			CqnListValue eTag = optETag.get().valueSet().asList();
			List<String> values = new ArrayList<>(eTag.size());
			eTag.values().forEach(v -> {
				if (v.isLiteral()) {
					values.add(v.asLiteral().value().toString());
				} else if (v.isParameter()) {
					values.add(context.getParameterValue(v.asParameter()).toString());
				}
			});
			return values;
		}
		if (entry instanceof CdsData cds) {
			Object entryETag = cds.getMetadata(METADATA_ETAG_KEY);
			if (entryETag != null) {
				return List.of(entryETag.toString());
			}
		}
		return Collections.emptyList();
	}

}
