/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.uri;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.impl.odata.utils.ConversionContext;
import com.sap.cds.services.impl.odata.utils.CqnToCloudSdkConverter;
import com.sap.cds.services.impl.odata.utils.ODataTypeUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ODataResourcePath;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataEntityKey;

public class UriGenerator {

	private final CqnAnalyzer analyzer;
	private final ConversionContext context;
	private final boolean operation;
	private boolean collection = true;

	public UriGenerator(ConversionContext context) {
		this(context, false);
	}

	public UriGenerator(ConversionContext context, boolean operation) {
		this.analyzer = CqnAnalyzer.create(context.getEventContext().getModel());
		this.context = context;
		this.operation = operation;
	}

	public ODataResourcePath analyze(CqnStatement statement) {
		return analyze(statement, Collections.emptyMap());
	}

	public ODataResourcePath analyze(CqnStatement statement, Map<String, Object> data) {
		if (statement == null) {
			return new ODataResourcePath();
		} else if (statement.isUpdate()) {
			return interpretAnalysis(analyzer.analyze(statement.asUpdate()), data, statement.asUpdate().where().isPresent());
		} else if (statement.isDelete()) {
			return interpretAnalysis(analyzer.analyze(statement.asDelete()), data, statement.asDelete().where().isPresent());
		} else if (statement.isSelect() && operation) {
			// for CqnSelect / GET we also only analyze the ref
			// as we prefer $filter on the collection for where conditions
			// for operations we also evaluate the where condition
			return interpretAnalysis(analyzer.analyze(statement.asSelect()), data, statement.asSelect().where().isPresent());
		} else {
			return interpretAnalysis(analyzer.analyze(statement.ref()), data, false);
		}
	}

	private ODataResourcePath interpretAnalysis(AnalysisResult analysis, Map<String, Object> data, boolean where) {
		ODataResourcePath uri = new ODataResourcePath();
		Iterator<ResolvedSegment> iterator = analysis.iterator();
		CdsEntity previous = null;
		while(iterator.hasNext()) {
			ResolvedSegment segment = iterator.next();
			if(previous == null) {
				uri.addSegment(segment.entity().getName());
				previous = segment.entity();
			} else {
				String navigationProperty = segment.segment().id();
				uri.addSegment(navigationProperty);
				collection = !CdsModelUtils.isSingleValued(previous.getAssociation(navigationProperty).getType());
				previous = previous.getTargetOf(navigationProperty);
			}

			boolean isTarget = !iterator.hasNext();
			if(segment.segment().filter().isPresent() || (isTarget && (where || !data.isEmpty()))) {
				Map<String, Object> keys = (isTarget && where) ? analysis.targetKeyValues() : segment.keyValues();
				if(isTarget && !data.isEmpty()) {
					segment.entity().keyElements()
						.map(CdsElement::getName)
						.filter(k -> !keys.containsKey(k)) // not specified by statement
						.filter(data::containsKey) // specified by data
						.forEach(k -> keys.put(k, data.get(k)));
				}
				if(keys.size() < segment.entity().keyElements().count()) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_INCOMPLETE_KEY, segment.entity().getQualifiedName());
				}
				uri.addParameterToLastSegment(ODataEntityKey.of(convertToCloudSdkTypedKeys(keys, segment.entity()), context.getProtocol()));
				collection = false;
			}
		}
		return uri;
	}

	private Map<String, Object> convertToCloudSdkTypedKeys(Map<String, Object> keys, CdsEntity entity) {
		Map<String, Object> cloudSdkTypedKeys = new HashMap<>();
		for(Map.Entry<String, Object> keyEntry : keys.entrySet()) {
			CdsBaseType keyType = ODataTypeUtils.getCdsType(entity, CQL.get(keyEntry.getKey()));
			cloudSdkTypedKeys.put(keyEntry.getKey(), CqnToCloudSdkConverter.convert(keyEntry.getValue(), entity, keyType, context));
		}
		return cloudSdkTypedKeys;
	}

	public boolean isCollection() {
		return collection;
	}

}
