/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.utils;

import static com.sap.cds.services.utils.model.CdsModelUtils.$USER;

import java.time.Instant;
import java.util.Locale;

import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnFunc;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions.Operand;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions.OperandSingle;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FieldReference;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FieldUntyped;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FilterExpressionArithmetic;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FilterExpressionString;
import com.sap.cloud.sdk.datamodel.odata.client.expression.FilterExpressionTemporal;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueBoolean;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueDate;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueDateTimeOffset;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueNumeric;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueString;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ValueTimeOfDay;

/**
 * Class to convert {@link CqnValue} objects to cloud sdk {@link OperandSingle} objects.
 */
public class CqnToCloudSdkConverter {

	private final Object left;
	private final Object right;
	private final CdsStructuredType structType;
	private final ConversionContext context;

	private OperandSingle operandLeft;
	private OperandSingle operandRight;
	private CdsBaseType type = null;

	public CqnToCloudSdkConverter(Object left, Object right, CdsStructuredType structType, ConversionContext context) {
		this.left = left;
		this.right = right;
		this.structType = structType;
		this.context = context;
	}

	public void convert() {
		operandLeft = toOperand(left, type);
		operandRight = toOperand(right, type);
		if (operandLeft == null) {
			operandLeft = toOperand(left, type, false);
		}
		if (operandRight == null) {
			operandRight = toOperand(right, type, false);
		}
	}

	public OperandSingle getLeft() {
		return operandLeft;
	}

	public OperandSingle getRight() {
		return operandRight;
	}

	/**
	 * Converts the given object to a {@link OperandSingle}
	 * @param obj the plain Java Object or {@link CqnValue} to convert
	 * @param structType the {@link CdsStructuredType}, required to resolve refs
	 * @param context the {@link ConversionContext}
	 *
	 * @return the converted value or {@code null} if the value could not be converted
	 */
	public static OperandSingle convert(Object obj, CdsStructuredType structType, ConversionContext context) {
		CqnToCloudSdkConverter converter = new CqnToCloudSdkConverter(obj, null, structType, context);
		converter.convert();
		return converter.getLeft();
	}

	/**
	 * Converts the given object to a {@link OperandSingle}
	 * @param obj the plain Java Object or {@link CqnValue} to convert
	 * @param structType the {@link CdsStructuredType}, required to resolve refs
	 * @param type the optional already known {@link CdsBaseType} of the object, used as an additional hint, when converting the object.
	 * @param context the {@link ConversionContext}
	 *
	 * @return the converted value or {@code null} if the value could not be converted
	 */
	public static OperandSingle convert(Object obj, CdsStructuredType structType, CdsBaseType type, ConversionContext context) {
		CqnToCloudSdkConverter converter = new CqnToCloudSdkConverter(obj, null, structType, context);
		converter.setType(type);
		converter.convert();
		return converter.getLeft();
	}

	/**
	 * Converts the given {@code value} by inferring type information from
	 * the {@code cqnValue}
	 * @param element the {@link CqnValue} to infer the type information from
	 * @param obj the value to convert
	 * @param structType the {@link CdsStructuredType}
	 * @param context the {@link ConversionContext}
	 * @return the converted value
	 */
	public static OperandSingle convert(Object obj, CdsStructuredType structType, CqnValue element, ConversionContext context) {
		CqnToCloudSdkConverter converter = new CqnToCloudSdkConverter(element, obj, structType, context);
		converter.convert();
		return converter.getRight();
	}

	private OperandSingle toFunction(CqnFunc func) {
		ODataProtocol protocol = context.getProtocol();
		try {
			switch(func.func().toLowerCase(Locale.US)) {
			// string functions
			case "contains":
				setType(CdsBaseType.BOOLEAN);
				if(protocol == ODataProtocol.V2) {
					// parameters in substringof are switched, in comparison to contains
					return FilterExpressionString.substringOf(toValueString(func.args().get(1)), toValueString(func.args().get(0)));
				}
				return FilterExpressionString.contains(toValueString(func.args().get(0)), toValueString(func.args().get(1)));
			case "endswith":
				setType(CdsBaseType.BOOLEAN);
				return FilterExpressionString.endsWith(toValueString(func.args().get(0)), toValueString(func.args().get(1)));
			case "startswith":
				setType(CdsBaseType.BOOLEAN);
				return FilterExpressionString.startsWith(toValueString(func.args().get(0)), toValueString(func.args().get(1)));
			case "length":
				setType(CdsBaseType.INTEGER);
				return FilterExpressionString.length(toValueString(func.args().get(0)));
			case "indexof":
				setType(CdsBaseType.INTEGER);
				return FilterExpressionString.indexOf(toValueString(func.args().get(0)), toValueString(func.args().get(1)));
			case "substring":
				setType(CdsBaseType.STRING);
				return FilterExpressionString.substring(toValueString(func.args().get(0)), toValueNumeric(func.args().get(1)));
			case "tolower":
				setType(CdsBaseType.STRING);
				return FilterExpressionString.toLower(toValueString(func.args().get(0)));
			case "toupper":
				setType(CdsBaseType.STRING);
				return FilterExpressionString.toUpper(toValueString(func.args().get(0)));
			case "trim":
				setType(CdsBaseType.STRING);
				return FilterExpressionString.trim(toValueString(func.args().get(0)));
			case "concat":
				setType(CdsBaseType.STRING);
				return FilterExpressionString.concat(toValueString(func.args().get(0)), toValueString(func.args().get(1)));
			case "matchespattern":
				return toMatchesPattern(func, protocol);
				// date functions
			case "year":
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.year(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.year(toValueDate(func.args().get(0)));
			case "month":
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.month(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.month(toValueDate(func.args().get(0)));
			case "day":
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.day(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.day(toValueDate(func.args().get(0)));
			case "hour":
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.hour(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.hour(toValueTimeOfDay(func.args().get(0)));
			case "minute":
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.minute(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.minute(toValueTimeOfDay(func.args().get(0)));
			case "second":
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.second(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.second(toValueTimeOfDay(func.args().get(0)));
			case "fractionalseconds":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}
				setType(CdsBaseType.INTEGER);
				if (isInstant(func.args().get(0))) {
					return FilterExpressionTemporal.fractionalSeconds(toValueDateTimeOffset(func.args().get(0)));
				}
				return FilterExpressionTemporal.fractionalSeconds(toValueTimeOfDay(func.args().get(0)));
			case "date":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}
				setType(CdsBaseType.DATE);
				return FilterExpressionTemporal.date(toValueDateTimeOffset(func.args().get(0)));
			case "time":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}

				setType(CdsBaseType.DATE);
				return FilterExpressionTemporal.time(toValueDateTimeOffset(func.args().get(0)));
			case "totaloffsetminutes":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}
				setType(CdsBaseType.INTEGER);
				return FilterExpressionTemporal.totalOffsetMinutes(toValueDateTimeOffset(func.args().get(0)));
			case "now":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}
				setType(CdsBaseType.TIMESTAMP);
				return FilterExpressionTemporal.now();
			case "mindatetime":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}
				setType(CdsBaseType.TIMESTAMP);
				return FilterExpressionTemporal.minDateTime();
			case "maxdatetime":
				if (protocol == ODataProtocol.V2) {
					throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
				}
				setType(CdsBaseType.TIMESTAMP);
				return FilterExpressionTemporal.maxDateTime();
				// math functions
			case "round":
				setType(CdsBaseType.INTEGER);
				return FilterExpressionArithmetic.round(toValueNumeric(func.args().get(0)));
			case "floor":
				setType(CdsBaseType.INTEGER);
				return FilterExpressionArithmetic.floor(toValueNumeric(func.args().get(0)));
			case "ceiling":
				setType(CdsBaseType.INTEGER);
				return FilterExpressionArithmetic.ceiling(toValueNumeric(func.args().get(0)));
			default:
				throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION, func.func());
			}
		} catch (IndexOutOfBoundsException e) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_NUM_ARGS, func.func(), e);
		}
	}

	private ValueString toValueString(CqnValue value) {
		OperandSingle result = toOperand(value, CdsBaseType.STRING);
		if (result instanceof FieldUntyped untyped) {
			return untyped.asString();
		}
		return (ValueString) result;
	}

	private ValueNumeric toValueNumeric(CqnValue value) {
		OperandSingle result = toOperand(value, null, false);
		if (result instanceof FieldUntyped untyped) {
			return untyped.asNumber();
		}
		return (ValueNumeric) result;
	}

	private ValueDate toValueDate(CqnValue value) {
		OperandSingle result = toOperand(value, CdsBaseType.DATE);
		if (result instanceof FieldUntyped untyped) {
			return untyped.asDate();
		}
		return (ValueDate) result;
	}

	private ValueDateTimeOffset toValueDateTimeOffset(CqnValue value) {
		OperandSingle result = toOperand(value, CdsBaseType.TIMESTAMP);
		if (result instanceof FieldUntyped untyped) {
			return untyped.asDateTimeOffset();
		}
		return (ValueDateTimeOffset) result;
	}

	private ValueTimeOfDay toValueTimeOfDay(CqnValue value) {
		OperandSingle result = toOperand(value, CdsBaseType.TIME);
		if (result instanceof FieldUntyped untyped) {
			return untyped.asTimeOfDay();
		}
		return (ValueTimeOfDay) result;
	}

	private OperandSingle toOdataLiteral(Object obj, CdsBaseType type) {
		ODataProtocol protocol = context.getProtocol();
		return Expressions.createOperand(ODataTypeUtils.toCloudSdkType(obj, type, protocol));
	}

	private OperandSingle handleRef(CqnElementRef ref) {
		EventContext eventContext = context.getEventContext();
		if(ref.firstSegment().equals("$now") && ref.size() == 1) {
			setType(CdsBaseType.TIMESTAMP);
			// TODO should we be able to get the NOW time stamp from the context?
			return toOdataLiteral(Instant.now(), type);
		}

		if (ref.size() == 2) {
			if ("$valid".equals(ref.firstSegment()) || "$at".equals(ref.firstSegment())) {
				if (ref.lastSegment().equals("from")) {
					setType(CdsBaseType.TIMESTAMP);
					return toOdataLiteral(eventContext.getParameterInfo().getValidFrom(), type);
				} else if (ref.lastSegment().equals("to")) {
					setType(CdsBaseType.TIMESTAMP);
					return toOdataLiteral(eventContext.getParameterInfo().getValidTo(), type);
				}
			}
	
			if (ref.firstSegment().equals($USER)) {
				if (ref.lastSegment().equals("locale")) {
					setType(CdsBaseType.STRING);
					return toOdataLiteral(LocaleUtils.getLocaleString(eventContext.getParameterInfo().getLocale()), type);
				} else if (ref.lastSegment().equals("id")) {
					setType(CdsBaseType.STRING);
					return toOdataLiteral(eventContext.getUserInfo().getName(), type);
				}
			}
		}

		if (ref.firstSegment().equals("$key") && ref.size() == 1) {
			if(structType.keyElements().count() == 1) {
				ref = CQL.get(structType.keyElements().findFirst().get().getName());
			}
		}

		setType(ODataTypeUtils.getCdsType(structType, ref));
		return convertRefToFieldUntyped(ref);
	}

	private OperandSingle handleParameter(CqnParameter parameter) {
		Object parameterValue = context.getParameterValue(parameter);
		return convert(parameterValue, structType, type, context);
	}

	private OperandSingle odataNull() {
		return Operand.NULL::getExpression;
	}

	private OperandSingle toOperand(Object obj, CdsBaseType type) {
		return toOperand(obj, type, true);
	}

	private OperandSingle toOperand(Object obj, CdsBaseType type, boolean requireType) {
		if (obj == null) {
			return odataNull();
		} else if (obj instanceof CqnValue value) {
			if (value.isRef()) {
				return handleRef(value.asRef());
			} else if (value.isFunction()) {
				return toFunction(value.asFunction());
			} else if (value.isLiteral() && (type != null || !requireType)) {
				return toOdataLiteral(value.asLiteral().value(), type);
			} else if (value.isNullValue()) {
				return odataNull();
			} else if (value.isParameter()) {
				return handleParameter(value.asParameter());
			}
		} else {
			return toOdataLiteral(obj, type);
		}
		return null;
	}

	private void setType(CdsBaseType type) {
		if (this.type == null) {
			this.type = type;
		}
	}

	public static FieldUntyped convertRefToFieldUntyped(CqnReference ref) {
		return FieldReference.ofPath(ref.segments().stream().map(s -> s.id()).toArray(String[]::new));
	}

	private static boolean isInstant(CqnValue value) {
		return value.isLiteral() && value.asLiteral().value() instanceof Instant;
	}

	private ValueBoolean toMatchesPattern(CqnFunc func, ODataProtocol protocol) {
		if(protocol == ODataProtocol.V2) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_FUNCTION_VERSION, func.func(), protocol);
		}
		setType(CdsBaseType.STRING);
		// Third argument is optional by the OData spec
		if (func.args().size() > 2 && !func.args().get(2).asLiteral().asString().value().isBlank()) {
			return new ExtendedMatchesPattern(func.args().get(0).asRef(), func.args().get(1), func.args().get(2));
		} else {
			return toValueString(func.args().get(0)).matches(toValueString(func.args().get(1)));
		}
	}
}
