/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.utils;

import static com.sap.cds.services.impl.odata.uri.ETagExtractor.METADATA_ETAG_KEY;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.http.Header;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.CdsData;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.DataUtils;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.request.NumberDeserializationStrategy;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestResultGeneric;

public class ODataDataUtils {

	private static final String ODATA_V2_RESULTS = "results";
	private static final String ODATA_V2_VALUE = "value";

	/**
	 * Extracts an entity map from the OData result.
	 * OData V2 and V4 responses are normalized by mapping the response to the entity structure (e.g. removing results substructures).
	 * Also non entity keys and values are removed (e.g. _metadata or @odata.etag)
	 *
	 * @param result the OData result
	 * @param resultType the structured type of the result
	 * @param protocol the OData protocol version
	 * @return the entity data
	 */
	public static CdsData entity(ODataRequestResultGeneric result, CdsStructuredType resultType, ODataProtocol protocol) {
		return withETag(result, normalize(preciseNumbers(result).asMap(), resultType, protocol));
	}

	/**
	 * Copies the original updated data to provide as a result for a "204 No Content" response.
	 * Takes care of updating the returned ETag on the data map.
	 *
	 * @param result the OData result
	 * @param data the original data
	 * @return the copied data, with updated ETag
	 */
	public static CdsData noContent(ODataRequestResultGeneric result, Map<String, Object> data) {
		return withETag(result, DataUtils.copyMap(data));
	}

	private static CdsData withETag(ODataRequestResultGeneric result, CdsData data) {
		Header eTagHeader = result.getHttpResponse().getFirstHeader(HttpHeaders.ETAG);
		if (eTagHeader != null) {
			String eTag = eTagHeader.getValue();
			if (!StringUtils.isEmpty(eTag)) {
				data.putMetadata(METADATA_ETAG_KEY, eTag);
			}
		}
		return data;
	}

	/**
	 * Extracts a list of entity maps from the OData result.
	 * OData V2 and V4 responses are normalized by mapping the response to the entity structure (e.g. removing results substructures).
	 * Also non entity keys and values are removed (e.g. _metadata or @odata.etag)
	 *
	 * @param result the OData result
	 * @param resultType the structured type of the result
	 * @param protocol the OData protocol version
	 * @return the list of entity data
	 */
	public static List<CdsData> entityCollection(ODataRequestResultGeneric result, CdsStructuredType resultType, ODataProtocol protocol) {
		return normalize(preciseNumbers(result).asListOfMaps(), resultType, protocol);
	}

	/**
	 * Extracts the return value of an operation from the OData result.
	 * OData V2 and V4 responses are normalized by mapping the response to the entity structure (e.g. removing results substructures).
	 * Also non entity keys and values are removed (e.g. _metadata or @odata.etag)
	 *
	 * @param result the OData result
	 * @param targetType the type of the result
	 * @param protocol the OData protocol version
	 * @param operationName the name of the operation
	 * @return the operation return value
	 */
	@SuppressWarnings("unchecked")
	public static Object operation(ODataRequestResultGeneric result, CdsType targetType, ODataProtocol protocol, String operationName) {
		result = preciseNumbers(result);
		if (ODataTypeUtils.isStructuredType(targetType)) {
			CdsStructuredType type = targetType.as(CdsStructuredType.class);
			Map<String, Object> mapResult = result.asMap();
			if (mapResult.size() == 1 && mapResult.containsKey(operationName)) {
				mapResult = (Map<String, Object>) mapResult.get(operationName);
			}
			return ODataTypeUtils.toCdsTypes(type, normalize(mapResult, type, protocol));
		} else if (ODataTypeUtils.isArrayedStructuredType(targetType)) {
			CdsStructuredType type = targetType.as(CdsArrayedType.class).getItemsType().as(CdsStructuredType.class);
			return ODataTypeUtils.toCdsTypes(type, normalize(result.asListOfMaps(), type, protocol));
		} else if (ODataTypeUtils.isSimpleType(targetType)) {
			// TODO Refactor once cloud sdk provides a way to retrieve a primitive result
			if (protocol == ODataProtocol.V2) {
				return ODataTypeUtils.toCdsType(result.asMap().get(operationName), targetType.as(CdsSimpleType.class).getType());
			}
			return ODataTypeUtils.toCdsType(result.asMap().get(ODATA_V2_VALUE), targetType.as(CdsSimpleType.class).getType());
		} else if (ODataTypeUtils.isArrayedSimpleType(targetType)) {
			CdsBaseType type = targetType.as(CdsArrayedType.class).getItemsType().as(CdsSimpleType.class).getType();
			// TODO Refactor once cloud sdk provides a way to retrieve a primitive result
			if (protocol == ODataProtocol.V2) {
				return ODataTypeUtils.toCdsTypes(type, (Iterable<Object>) result.asMap().get(ODATA_V2_RESULTS));
			}
			return ODataTypeUtils.toCdsTypes(type, (Iterable<Object>) result.asMap().get(ODATA_V2_VALUE));
		}
		return result;
	}

	/**
	 * Cleans unknown elements from a list of entity data.
	 * All keys in the map that can't be associated with an entity element are removed.
	 *
	 * @param entryData the incoming data
	 * @param entryType the structured type of the data
	 */
	public static void clean(List<Map<String, Object>> entryData, CdsStructuredType entryType) {
		DataProcessor.create()
		.bulkAction((type, entries) -> {
			Set<String> elements = type.elements().map(CdsElement::getName).collect(Collectors.toSet());
			entries.forEach(entry -> entry.entrySet().removeIf(e -> !elements.contains(e.getKey())));
		})
		.process(entryData, entryType);
	}

	@VisibleForTesting
	static CdsData normalize(Map<String, Object> data, CdsStructuredType dataType, ODataProtocol protocol) {
		return normalize(Arrays.asList(data), dataType, protocol).get(0);
	}

	@VisibleForTesting
	static List<CdsData> normalize(List<Map<String, Object>> list, CdsStructuredType dataType, ODataProtocol protocol) {
		List<CdsData> data = DataUtils.copyGenericList(list);
		DataProcessor.create().action((currentStructType, r) -> {
			CdsData row = (CdsData) r;
			for (String key: new HashSet<>(row.keySet())) {
				Optional<CdsElement> element = currentStructType.findElement(key);
				if (element.isPresent()) {
					CdsType elementType = element.get().getType();
					if(protocol == ODataProtocol.V2 && elementType.isAssociation() && row.get(key) instanceof Map) {
						Map<?, ?> assocValue = (Map<?,?>) row.get(key);
						// to one association
						if(CdsModelUtils.isSingleValued(elementType)) {
							CdsStructuredType assocTarget = elementType.as(CdsAssociationType.class).getTarget();
							if (!assocValue.isEmpty() && !assocValue.keySet().stream().anyMatch(k -> assocTarget.elements().anyMatch(e -> Objects.equals(e.getName(), k)))) {
								// not a single property of the assoc map matches actual structured type elements
								// only metadata was provided, but no actual payload, therefore we store null
								row.put(key, null);
							}
						}
						// to-many association
						else {
							// in OData V2 expanded to many entities are stored in the "results" field
							// if that is not available we store null
							row.put(key, assocValue.get(ODATA_V2_RESULTS));
						}

					}
				// TODO what about other metadata values?
				} else if (key.equals("@etag") || key.equals("@odata.etag")) {
					row.putMetadata(METADATA_ETAG_KEY, row.remove(key));
				} else if (key.equals("__metadata") && row.get(key) instanceof Map metadataMap && metadataMap.containsKey("etag")) {
					row.putMetadata(METADATA_ETAG_KEY, metadataMap.get("etag"));
					row.remove(key);
				} else {
					row.remove(key);
				}
			}
		}).process(data, dataType);
		return data;
	}

	private static ODataRequestResultGeneric preciseNumbers(ODataRequestResultGeneric result) {
		return result.withNumberDeserializationStrategy(NumberDeserializationStrategy.BIG_DECIMAL);
	}

}
