/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.utils;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;

import org.apache.http.HttpHeaders;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.environment.CdsProperties.Remote.RemoteServiceConfig.Http;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.QueryParameters;
import com.sap.cloud.sdk.cloudplatform.connectivity.CsrfTokenRetriever;
import com.sap.cloud.sdk.datamodel.odata.client.query.StructuredQuery;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataRequestGeneric;
import com.sap.cloud.sdk.datamodel.odata.client.request.ODataUriFactory;

public class ODataRequestUtils {

	public static void prepareQuery(StructuredQuery query, EventContext context) {
		ParameterInfo info = context.getParameterInfo();

		if (info.getValidFrom() != null && info.getValidTo() != null && info.getValidFrom().equals(info.getValidTo().minusMillis(1))) {
			query.withCustomParameter(QueryParameters.VALID_AT, format(info.getValidFrom()));
		} else {
			if (info.getValidFrom() != null) {
				query.withCustomParameter(QueryParameters.VALID_FROM, format(info.getValidFrom()));
			}
			if (info.getValidTo() != null) {
				query.withCustomParameter(QueryParameters.VALID_TO, format(info.getValidTo()));
			}
		}
		if (info.getLocale() != null) {
			query.withCustomParameter(QueryParameters.SAP_LANGUAGE, LocaleUtils.getLocaleStringForQuery(info.getLocale()));
		}
	}

	public static <T extends ODataRequestGeneric> T prepareRequest(T req, Http httpConfig, EventContext context) {
		ParameterInfo info = context.getParameterInfo();
		if (info.getLocale() != null) {
			req.addHeader(HttpHeaders.ACCEPT_LANGUAGE, info.getLocale().toLanguageTag());
		}
		httpConfig.getHeaders().forEach(req::addHeader);
		httpConfig.getQueries().forEach((k, v) -> req.addQueryParameter(k, ODataUriFactory.encodeQuery(v)));
		if (!httpConfig.getCsrf().isEnabled()) {
			req.setCsrfTokenRetriever(CsrfTokenRetriever.DISABLED_CSRF_TOKEN_RETRIEVER);
		}
		return req;
	}

	private static String format(Instant instant) {
		return DateTimeFormatter.ISO_OFFSET_DATE_TIME.format(instant.atZone(ZoneId.of("UTC")));
	}

}
