/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.odata.utils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import org.apache.commons.codec.binary.Base64;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsTypeUtils;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

public class ODataTypeUtils {

	public static final long FACTOR_MILLIS_TO_DAYS = 1000*60*60*24;

	/**
	 * Converts the {@link Object} to the correct Java type to be used with Cloud SDK.
	 * @param value the value to convert
	 * @param type the {@link CdsBaseType}
	 * @param protocol the {@link ODataProtocol}
	 * @return the {@code value} converted to the correct type
	 */
	public static Object toCloudSdkType(Object value, CdsBaseType type, ODataProtocol protocol) {
		if(value == null) {
			return null;
		}

		if (type != null) {
			switch (type) {
			case UUID:
				return UUID.fromString((String) value);
			case STRING:
				return value.toString();
			default:
				break;
			}
		}
		// there is no Date type in OData V2, instead DateTime is used
		if (protocol == ODataProtocol.V2 && value instanceof LocalDate date) {
			return LocalDateTime.of(date, LocalTime.MIDNIGHT);
		}
		if (value instanceof Instant instant) {
			return OffsetDateTime.ofInstant(instant, ZoneOffset.UTC);
		}
		return value;
	}

	/**
	 * This method determines the CDS type of the reference {@code ref} belonging to the {@code structuredType}
	 * @param structuredType the {@link CdsStructuredType}
	 * @param ref the {@link CqnElementRef}
	 * @return the CDS type if it could be determined
	 */
	public static CdsBaseType getCdsType(CdsStructuredType structuredType, CqnElementRef ref) {
		CdsElement element = null;
		for (Segment segment: ref.segments()) {
			element = structuredType.getElement(segment.id());
			if (element.getType().isAssociation()) {
				structuredType = structuredType.getTargetOf(element.getName());
			}
		}

		if (element != null) {
			if (element.getType().isSimple()) {
				// enums are also simple types
				return element.getType().as(CdsSimpleType.class).getType();
			} else if (element.getType().isArrayed()) {
				CdsType type = element.getType();
				while (type.isArrayed()) {
					type = type.as(CdsArrayedType.class).getItemsType();
				}
				if (type.isSimple()) {
					return type.as(CdsSimpleType.class).getType();
				}
			}
		}

		throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_REF_TYPE, ref.path(), structuredType.getQualifiedName());
	}

	public static <T extends Map<String, Object>> T toCdsTypes(CdsStructuredType structuredType, T data) {
		return toCdsTypes(structuredType, Arrays.asList(data)).get(0);
	}

	/**
	 * Converts the values in {@code data} to the correct Java types
	 * according to the respective {@link CdsBaseType} defined in the CDS model.
	 *
	 * @param structuredType the {@link CdsStructuredType}
	 * @param data the data to convert
	 * @param <T> the data type
	 * @return The converted data
	 */
	public static <T extends Iterable<? extends Map<String, Object>>> T toCdsTypes(CdsStructuredType structuredType, T data) {
		DataProcessor.create().action((currentStructType, row) -> {
			for (String key: new HashSet<>(row.keySet())) {
				Optional<CdsElement> element = currentStructType.findElement(key);
				if (element.isPresent()) {
					CdsType elementType = element.get().getType();
					if (elementType.isSimple()) {
						row.put(key, toCdsType(row.get(key), elementType.as(CdsSimpleType.class).getType()));
					}
				}
			}
		}).process(data, structuredType);
		return data;
	}

	public static Iterable<?> toCdsTypes(CdsBaseType type, Iterable<? extends Object> data) {
		List<Object> result = new ArrayList<>();
		data.forEach(e -> result.add(toCdsType(e, type)));
		return result;
	}

	/**
	 * This method converts the {@code value} to the correct Java type defined in {@code type}.
	 * @param value the value to convert
	 * @param type the {@link CdsBaseType} to convert to
	 * @return the converted value
	 * @throws ErrorStatusException in case the value cannot be converted
	 */
	@SuppressWarnings("deprecation")
	public static Object toCdsType(Object value, CdsBaseType type) {
		if (value == null) {
			return value;
		}
		try {
			if (type.javaType().isInstance(value)) {
				return value;
			}
			switch (type) {
			case DECIMAL:
			case DECIMAL_FLOAT:
				if (value instanceof Double || value instanceof Float) {
					return new BigDecimal(handlePrimitiveTypesDouble(value));
				} else if (value instanceof Long || value instanceof Integer || value instanceof Short || value instanceof Byte) {
					return new BigDecimal(handlePrimitiveTypesLong(value));
				} else if (value instanceof BigInteger integer) {
					return new BigDecimal(integer);
				}
				break;
			case DOUBLE:
				if (value instanceof BigDecimal decimal) {
					return decimal.doubleValue();
				} else if (value instanceof BigInteger integer) {
					return integer.doubleValue();
				} else if (value instanceof Number) {
					return handlePrimitiveTypesDouble(value);
				}
				break;
			case INTEGER:
				if (value instanceof BigDecimal decimal) {
					return decimal.intValue();
				} else if (value instanceof BigInteger integer) {
					return integer.intValue();
				} else if (value instanceof Number) {
					return (int) handlePrimitiveTypesLong(value);
				}
				break;
			case INTEGER64:
				if (value instanceof BigDecimal decimal) {
					return decimal.longValue();
				} else if (value instanceof BigInteger integer) {
					return integer.longValue();
				} else if (value instanceof Number) {
					return handlePrimitiveTypesLong(value);
				}
				break;
			case DATETIME:
			case TIMESTAMP:
				if (value instanceof LocalDateTime time) {
					return time.toInstant(ZoneOffset.UTC);
				} else if (value instanceof OffsetDateTime time) {
					return time.toInstant();
				} else if (value instanceof LocalDate date) {
					return date.atStartOfDay().toInstant(ZoneOffset.UTC);
				} else if (isJsDate(value)) {
					return Instant.ofEpochMilli(jsDateToEpochMillis((String) value));
				}
				break;
			case DATE:
				if (value instanceof LocalDateTime time) {
					return time.toLocalDate();
				} else if (value instanceof OffsetDateTime time) {
					return time.toLocalDate();
				} else if (isJsDate(value)) {
					long epochMillis = jsDateToEpochMillis((String) value);
					return LocalDate.ofEpochDay(epochMillis/FACTOR_MILLIS_TO_DAYS);
				}
				break;
			case TIME:
				if (value instanceof LocalDateTime time) {
					return time.toLocalTime();
				} else if (value instanceof OffsetDateTime time) {
					return time.toLocalTime();
				} else if (value instanceof String string && string.startsWith("P")) {
					Duration duration = Duration.parse(string);
					return LocalTime.of(0, 0).plus(duration);
				}
				break;
			case UUID:
				if (value instanceof UUID) {
					return value.toString();
				}
				break;
			case STRING:
			case LARGE_STRING:
				return value.toString();
			case BINARY:
			case LARGE_BINARY:
				if (value instanceof String string) {
					// Apache's Base64 is more robust than java.util
					return Base64.decodeBase64(string);
				}
				break;
			default:
				break;
			}
			return CdsTypeUtils.parse(type, value.toString());
		} catch (ClassCastException | CdsDataException e) {
			throw new ErrorStatusException(CdsErrorStatuses.REMOTE_ODATA_TYPE_CONV, value, value.getClass(), type, e);
		}
	}

	private static boolean isJsDate(Object value) {
		return value instanceof String s && s.startsWith("/Date(");
	}

	private static long jsDateToEpochMillis(String jsDate) {
		String time = jsDate.substring(6, jsDate.length() - 2);
		// minus and plus as the first character are not interpreted as offset start
		int zoneStart = time.lastIndexOf("+");
		if (zoneStart < 1) {
			zoneStart = time.lastIndexOf("-");
		}
		return Long.parseLong(time.substring(0, zoneStart > 0 ? zoneStart : time.length()));
	}

	/**
	 * Helper method for primitive type handling
	 * @param value the value
	 * @return the value properly casted to {@code long}
	 */
	private static long handlePrimitiveTypesLong(Object value) {
		if (value instanceof Byte byte1) {
			return byte1.longValue();
		} else if (value instanceof Short short1) {
			return short1.longValue();
		} else if (value instanceof Integer integer) {
			return integer.longValue();
		} else if (value instanceof Float float1) {
			return float1.longValue();
		} else if (value instanceof Double double1) {
			return double1.longValue();
		}
		return (Long) value;
	}

	/**
	 * Helper method for primitive type handling
	 * @param value the value
	 * @return the value properly casted to {@code double}
	 */
	private static double handlePrimitiveTypesDouble(Object value) {
		if (value instanceof Byte byte1) {
			return byte1.doubleValue();
		} else if (value instanceof Short short1) {
			return short1.doubleValue();
		} else if (value instanceof Integer integer) {
			return integer.doubleValue();
		} else if (value instanceof Long long1) {
			return long1.doubleValue();
		} else if (value instanceof Float float1) {
			return float1.doubleValue();
		}
		return (Double) value;
	}

	public static boolean isSimpleType(CdsType type) {
		return type != null && type.isSimple();
	}

	public static boolean isArrayedSimpleType(CdsType type) {
		return type != null && type.isArrayed() && type.as(CdsArrayedType.class).getItemsType().isSimple();
	}

	public static boolean isStructuredType(CdsType type) {
		return type != null && type.isStructured();
	}

	public static boolean isArrayedStructuredType(CdsType type) {
		return type != null && type.isArrayed() && type.as(CdsArrayedType.class).getItemsType().isStructured();
	}

}
