/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.actuator;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.boot.actuate.endpoint.annotation.Selector;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.server.ResponseStatusException;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.info.CdsInfo;
import com.sap.cds.services.utils.info.CdsInfo.Details;

@Component
@ConditionalOnClass(Endpoint.class)
@Endpoint(id = "cds", enableByDefault = true)
public class CdsActuatorEndpoint {

	@Value("#{'${management.endpoint.cds.details:MEDIUM}'.toUpperCase()}")
	private Details defaultDetails;

	private final List<CdsInfo> cdsInfoList = new ArrayList<>();

	@Autowired
	public CdsActuatorEndpoint(CdsRuntime runtime) {
		ServiceLoader<CdsInfo> infoSectionLoader = ServiceLoader.load(CdsInfo.class);
		infoSectionLoader.iterator().forEachRemaining(cdsInfo -> {
			cdsInfo.setRuntime(runtime);
			cdsInfoList.add(cdsInfo);
		});
	}

	@ReadOperation
	public Map<String, Object> info(@RequestParam @org.springframework.lang.Nullable String details) {
		return createCdsInfo(i -> true, gainDetails(details)); // NOSONAR
	}

	@ReadOperation
	public Map<String, Object> infoEntry(@Selector String info, @RequestParam @org.springframework.lang.Nullable String details) {
		return createCdsInfo(i -> i.name().equalsIgnoreCase(info), gainDetails(details)); // NOSONAR
	}

	private Details gainDetails(String details) {
		if (!StringUtils.isEmpty(details)) {
			try {
				return Details.valueOf(details.toUpperCase(Locale.ENGLISH));
			} catch(Exception e) { // NOSONAR
				throw new ResponseStatusException(HttpStatus.BAD_REQUEST, "Invalid detail parameter '" + details + "'");
			}
		}
		return defaultDetails;
	}

	private Map<String, Object> createCdsInfo(Predicate<CdsInfo> filter, Details details) {
		Map<String, Object> cdsInfo = new LinkedHashMap<>();

		cdsInfoList.stream().filter(filter).forEach(section -> {
			cdsInfo.put(section.name(), section.info(details));
		});

		return cdsInfo;
	}
}
