/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.actuator;

import javax.sql.DataSource;

import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.actuate.jdbc.DataSourceHealthIndicator;

import com.sap.cds.feature.config.Properties;
import com.sap.cloud.mt.runtime.DbHealthIndicatorImpl;
import com.sap.cloud.mt.runtime.TenantAwareDataSource;

/**
 * A {@link HealthIndicator} for DB availability
 * which can be used in MT scenario.
 */
public class DbHealthIndicatorMt implements HealthIndicator {

	private DbHealthIndicatorImpl<Health> dbHealthIndicatorMt;

	private DataSourceHealthIndicator dbHealthIndicatorSt;

	public DbHealthIndicatorMt(DataSource dataSource) {
		long intervallMillis = Properties.getCds().getMultiTenancy().getHealthCheck().getIntervalMillis();
		String healthCheckStatement = Properties.getCds().getMultiTenancy().getHealthCheck().getHealthCheckStatement();
		boolean newCheck = !Properties.getCds().getMultiTenancy().getDataSource().getHanaDatabaseIds().isEmpty();

		if (dataSource instanceof TenantAwareDataSource) { // Note that even if MT is configured thoroughly, the primary DataSource may be overridden (e.g. local db)
			this.dbHealthIndicatorMt = new DbHealthIndicatorImpl<Health>(healthCheckStatement, (TenantAwareDataSource)dataSource, intervallMillis,
					() -> Health.up().build(),
					() -> Health.down().build(),
					(text, details) -> Health.down().withDetail(text, details).build(),
					(text, details) -> Health.up().withDetail(text, details).build(),
					newCheck);
		} else {
			this.dbHealthIndicatorSt = new DataSourceHealthIndicator(dataSource);
		}
	}

	@Override
	public Health health() {
		if (dbHealthIndicatorMt != null) {
			return dbHealthIndicatorMt.health();
		} else {
			return dbHealthIndicatorSt.health();
		}
	}

}
