/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.utils;

import static com.sap.cds.services.utils.mtx.MtxUtils.getBearerAuthorizationHeader;

import java.util.Arrays;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.builder.SpringApplicationBuilder;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.config.pojo.CdsProperties.MultiTenancy;
import com.sap.cds.mtx.impl.ClientCredentialJwtAccess;
import com.sap.cds.services.mt.MtSubscriptionService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.mtx.MtxUtils;
import com.sap.cloud.mt.subscription.json.SidecarUpgradePayload;

/**
 * Class providing a main method to update the database content.
 * This main method starts the whole server. Therefore, all background tasks that might be registered will also start.
 * If this could lead to problems, they need to be deactivated.
 */
@SpringBootApplication
public class Deploy {

	private static final Logger log = LoggerFactory.getLogger(Deploy.class);
	private static final ObjectMapper mapper = new ObjectMapper();

	private static final String SIDECAR_STATUS_RUNNING = "RUNNING";

	private static final String SIDECAR_RESPONSE_FIELD_STATUS = "status";
	private static final String SIDECAR_RESPONSE_FIELD_ERROR = "error";
	private static final String SIDECAR_RESPONSE_FIELD_JOB_ID = "jobID";

	private static final String PARAMETER_ALL_TENANTS = "all";

	public static void main(String[] args) throws Exception {
		try {
			// deactivate authentication and messaging
			System.setProperty("cds.security.xsuaa.enabled", "false");
			System.setProperty("cds.security.mock.enabled", "false");
			System.setProperty("cds.messaging.receiver.enabled", "false");

			// start without web server
			CdsRuntime runtime = new SpringApplicationBuilder(Deploy.class).web(WebApplicationType.NONE).run(new String[0]).getBean(CdsRuntime.class);
			MtSubscriptionService mtService = runtime.getServiceCatalog().getService(MtSubscriptionService.class, MtSubscriptionService.DEFAULT_NAME);
			// make sure the property names above stay correct
			assert !Properties.getCds().getSecurity().getXsuaa().isEnabled();
			assert !Properties.getCds().getSecurity().getMock().isEnabled();
			assert !Properties.getCds().getMessaging().getReceiver().isEnabled();

			if (mtService != null) {
				long timestamp = System.currentTimeMillis();
				String[] tenants = Arrays.copyOf(args, args.length);
				if (tenants.length == 0) {
					tenants = new String[] {PARAMETER_ALL_TENANTS};
					log.info("Starting database update for all tenants");
				} else {
					log.info("Starting database update for tenant(s) {}", String.join(", ", tenants));
				}
				SidecarUpgradePayload upgradePayload = new SidecarUpgradePayload();
				upgradePayload.tenants = tenants;

				MultiTenancy config = Properties.getCds().getMultiTenancy();
				String deployScope = config.getSecurity().getDeploymentScope();

				runtime.requestContext().clearUser().modifyUser(user -> user.addRole(deployScope).setIsAuthenticated(true)).run(context -> {
					if (MtxUtils.mtxEnabled()) {
						try {
							// obtains a JWT token through client credential flow
							ClientCredentialJwtAccess jwtAccess = new ClientCredentialJwtAccess(MtxUtils.createClientCredentialJwtReader());
							String jobId = startDbDeploymentSidecar(upgradePayload, mtService, jwtAccess);
							waitForDeploymentToFinish(jobId, mtService, jwtAccess);
						} catch (Throwable t) {
							throw new RuntimeException(t);
						}
					} else {
						mtService.deploy(upgradePayload);
					}
					return null;
				});
				log.info("Database update finished successfully in {}s", (System.currentTimeMillis() - timestamp)/1000);
			} else {
				log.error("Failed: MT Service not found");
				logErrorMessage();
				System.exit(2);
			}
		} catch (Throwable t) {
			log.error("Unexpected error", t);
			logErrorMessage();
			System.exit(1);
		}
		logSuccessMessage();
		System.exit(0);
	}

	private static String startDbDeploymentSidecar(SidecarUpgradePayload upgradePayload, MtSubscriptionService mtService,
			ClientCredentialJwtAccess jwtAccess) throws Exception {
		String jobIdResultStr = mtService.asyncDeploy(upgradePayload, getBearerAuthorizationHeader(jwtAccess.getJwt()));

		TypeReference<Map<String, String>> typeRefStr = new TypeReference<Map<String, String>>() {};

		Map<String, String> jobIdResult = mapper.readValue(jobIdResultStr, typeRefStr);
		return jobIdResult.get(SIDECAR_RESPONSE_FIELD_JOB_ID);
	}

	private static void waitForDeploymentToFinish(String jobId, MtSubscriptionService mtService, ClientCredentialJwtAccess jwtAccess) throws Exception {
		TypeReference<Map<String, Object>> typeRef = new TypeReference<Map<String, Object>>() {};
		String resultStr;
		String status;
		Map<String, Object> result;

		long lastTimestamp = 0;

		do {
			// print out status once a minute
			if (System.currentTimeMillis() - lastTimestamp > 60000) {
				log.info("Waiting for database update to finish...");
				lastTimestamp = System.currentTimeMillis();
			}

			Thread.sleep(2000);

			resultStr = mtService.asyncDeployStatus(jobId, getBearerAuthorizationHeader(jwtAccess.getJwt()));
			result = mapper.readValue(resultStr, typeRef);
			status = (String) result.get(SIDECAR_RESPONSE_FIELD_STATUS);
		} while (SIDECAR_STATUS_RUNNING.equals(status));
		if (result.get(SIDECAR_RESPONSE_FIELD_ERROR) != null) {
			log.error("Database update failed, last sidecar response: {}", resultStr);
			logErrorMessage();
			System.exit(3);
		}
		log.debug("Last sidecar response: {}", resultStr);
	}

	private static void logSuccessMessage() {
		log.info("*************");
		log.info("*  SUCCESS  *");
		log.info("*************");
	}

	private static void logErrorMessage() {
		log.error("***********");
		log.error("*  ERROR  *");
		log.error("***********");
	}

}
