/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth;

import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.ExpressionUrlAuthorizationConfigurer;

import com.sap.cds.adapter.ServletUrlResourcePaths;
import com.sap.cds.adapter.ServletUrlResourcePaths.UrlResourcePathVisitor;
import com.sap.cds.adapter.UrlResourcePath;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * A {@link ConfigureServletAdapterSecurity}  configures the authentication of the endpoints
 * exposed by the registered Servlet adapters.
 * Service endpoints that require no authorization as defined in the CDS model are exposed as public.
 */
public class ConfigureServletAdapterSecurity {

	private final static Logger logger = LoggerFactory.getLogger(ConfigureServletAdapterSecurity.class);

	public static void configure(CdsRuntime cdsRuntime, HttpSecurity http) throws Exception { // NOSONAR

		ServletUrlResourcePaths servletPaths = new ServletUrlResourcePaths(cdsRuntime.getCdsModel());

		if( !Properties.getCds().getSecurity().isAuthenticateUnknownEndpoints() ) {
			// narrow the base urls to the urls of the servlets
			http.requestMatchers().antMatchers(servletPaths.getBasePaths().map(UrlResourcePath::getPath).toArray(String[]::new));
			logger.info("Configuring authentication of CDS adapter endpoints. Other endpoints are not configured.");
		}

		ExpressionUrlAuthorizationConfigurer<HttpSecurity>.ExpressionInterceptUrlRegistry urlRegistry = http.authorizeRequests();
		if ( Properties.getCds().getSecurity().isOpenUnrestrictedEndpoints() ) {
			// open endpoints which are declared as public
			servletPaths.visit(new UrlResourcePathVisitor() {

				@Override
				public void foundPublicPath(UrlResourcePath publicPath) {
					urlRegistry.antMatchers(publicPath.getPath()).permitAll();
					logger.debug("Public CDS endpoint {}", publicPath.getPath());
				}

				@Override
				public void foundPublicEvents(UrlResourcePath path, Stream<String> publicEvents) {
					publicEvents.forEach( publicEvent -> {
						urlRegistry.antMatchers(HttpMethod.valueOf(publicEvent), path.getPath()).permitAll();
						logger.debug("Public CDS endpoint {} {}", publicEvent, path.getPath());
					});

					urlRegistry.antMatchers(path.getPath()).authenticated(); // Required as parent path may be open with recursive pattern
					logger.debug("Authenticate CDS endpoint {}", path.getPath());
				}

			});
		} else {
			logger.info("Disabled configuration of public CDS adapter endpoints. All CDS adapter endpoints require authentication.");
		}

		// all other endpoints of the servlets are closed by default
		urlRegistry.anyRequest().authenticated();
	}
}
