/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.oauth2.jwt.Jwt;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cloud.security.xsuaa.XsuaaServiceConfiguration;
import com.sap.cloud.security.xsuaa.token.TokenAuthenticationConverter;


/**
 * The default security configuration in case of active XSUAA configuration - which should be the case in productive scenarios.
 * By default, all requests require an oauth2 authentication - including public endpoints.
 * In order to open public endpoints, the application needs to create a security configuration with higher priority
 * and override these endpoints.
 */
@Configuration
@ConditionalOnClass({ XsuaaServiceConfiguration.class, WebSecurityConfigurer.class, Jwt.class }) // only if xsuaa and spring-security web & JWT library has been loaded
@Conditional(XsuaaConfiguredCondition.class)
@EnableWebSecurity
@Order(1000)
public class XsuaaSecurityConfig extends WebSecurityConfigurerAdapter {

	private final static Logger logger = LoggerFactory.getLogger(XsuaaSecurityConfig.class);

	@Autowired
	private XsuaaServiceConfiguration xsuaaServiceConfiguration;

	@Autowired
	private CdsRuntime cdsRuntime;

	@Override
	protected void configure(HttpSecurity http) throws Exception {

		// @formatter:off
		http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS); // state is kept in AppRouter

		ConfigureServletAdapterSecurity.configure(cdsRuntime, http);

		http.oauth2ResourceServer()
		.jwt()
		.jwtAuthenticationConverter(getJwtAuthoritiesConverter());
		// @formatter:on

		logger.info("Activated XSUAA security configuration based on OAuth 2");
	}

	Converter<Jwt, AbstractAuthenticationToken> getJwtAuthoritiesConverter() {
		TokenAuthenticationConverter converter = new TokenAuthenticationConverter(xsuaaServiceConfiguration);
		converter.setLocalScopeAsAuthorities(true);
		return converter;
	}

}
