/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collector;
import java.util.stream.Stream;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.runtime.ParameterInfoProvider;
import com.sap.cds.services.runtime.UserInfoProvider;

/**
 * Eagerly initialized singleton bean, that performs the initialization of the CdsRuntime.
 * Using a eagerly initialized singleton bean ensures, that this code is executed before a web-server (e.g. tomcat) is started.
 */
@Component
public class CdsRuntimeInitializer implements InitializingBean, ApplicationContextAware {

	@Autowired
	private CdsRuntimeConfigurer configurer;

	@Autowired
	private List<Service> services;

	@Autowired(required = false)
	private List<UserInfoProvider> userProviders;

	@Autowired(required = false)
	private List<ParameterInfoProvider> parameterProviders;

	@Autowired(required = false)
	private List<CdsModelProvider> modelProviders;

	private ApplicationContext context;

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.context = applicationContext;
	}

	@Override
	@SuppressWarnings({"unchecked", "rawtypes"})
	public void afterPropertiesSet() throws Exception {
		// ensure that all services are registered
		ServiceCatalog catalog = configurer.getCdsRuntime().getServiceCatalog();
		services.stream().filter(s -> catalog.getService(s.getName()) == null).forEach(configurer::service);

		// register event handlers
		configurer.eventHandlerConfigurations();

		// register custom event handlers
		Arrays.stream(context.getBeanNamesForType(EventHandler.class))
		// filter all beans which types have been duplicated through scoping by CGLIB
		.filter(name -> !name.startsWith("scopedTarget."))
		.map(context::getType).distinct().forEach((handlerClass) -> {
			configurer.eventHandler((Class) handlerClass, () -> context.getBean(handlerClass));
		});

		// register providers. The lists are in sorted order induced by @Order annotation.
		if (userProviders != null) {
			reverse(userProviders.stream()).forEach(configurer::provider);
		}

		if (parameterProviders != null) {
			reverse(parameterProviders.stream()).forEach(configurer::provider);
		}

		if (modelProviders != null) {
			reverse(modelProviders.stream()).forEach(configurer::provider);
		}

		// give chance to change providers in custom handlers
		catalog.getServices(ApplicationLifecycleService.class).forEach(as -> as.applicationProviders(configurer));

		// lock cds runtime configurer
		configurer.complete();

		// fire runtime started event - after all beans have been initialized and before web server is up and running
		catalog.getServices(ApplicationLifecycleService.class).forEach(as -> as.applicationPrepared());
	}

	/**
	 * Creates a reverse stream.
	 * @param stream	The stream to be put in reverse order
	 * @return	The stream in reverse order
	 */
	private static <T> Stream<T> reverse(Stream<T> stream) {
		return stream.collect(Collector.of(() -> new ArrayDeque<T>(),
				ArrayDeque::addFirst,
				(q1, q2) -> { q2.addAll(q1); return q2; }))
				.stream();
	}

}
