/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.function.Supplier;

import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.jdbc.DataSourceTransactionManagerAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.TransactionAwareDataSourceProxy;
import org.springframework.transaction.PlatformTransactionManager;

import com.sap.cds.CdsDataStore;
import com.sap.cds.framework.spring.transaction.PlatformTransactionManagerPostProcessor.SpringTransactionManagerGetter;
import com.sap.cds.framework.spring.utils.ProxyUtils;
import com.sap.cds.services.impl.persistence.JdbcPersistenceService;
import com.sap.cds.services.impl.persistence.JdbcPersistenceServiceConfiguration;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.transaction.TransactionManager;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

@Configuration
// only if cds-feature-jdbc, spring-jdbc and spring-tx are available
@ConditionalOnClass({ JdbcPersistenceServiceConfiguration.class, TransactionAwareDataSourceProxy.class, PlatformTransactionManager.class })
@ConditionalOnBean({ DataSource.class, PlatformTransactionManager.class }) // datasource and tx manager must exist
@AutoConfigureAfter({ DataSourceAutoConfiguration.class, DataSourceTransactionManagerAutoConfiguration.class})
public class SpringJdbcPersistenceServiceConfiguration {

	private final JdbcPersistenceServiceConfiguration configuration;

	@Autowired
	public SpringJdbcPersistenceServiceConfiguration(DataSource dataSource, PlatformTransactionManager platformTransactionManager) {
		TransactionManager txMgr = ((SpringTransactionManagerGetter) platformTransactionManager).getSpringTransactionManager();
		DataSource txAwareDataSource = new TransactionAwareDataSourceProxy(dataSource);
		Supplier<Connection> connectionSupplier = () -> {
			try {
				return txAwareDataSource.getConnection();
			} catch (SQLException e) {
				throw new ErrorStatusException(CdsErrorStatuses.JDBC_CONNECTION_FAILED, e);
			}
		};

		this.configuration = JdbcPersistenceServiceConfiguration.create(connectionSupplier, txMgr);
	}

	@Bean(PersistenceService.DEFAULT_NAME)
	public JdbcPersistenceService persistenceService(CdsRuntime runtime) {
		return configuration.getService(runtime);
	}

	@Bean
	public CdsDataStore cdsDataStore(CdsRuntime runtime) {
		JdbcPersistenceService persistenceService = configuration.getService(runtime);
		return ProxyUtils.createProxy(CdsDataStore.class, () -> persistenceService.getCdsDataStore());
	}

}
