/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.feature;

import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.authentication.OAuth2AuthenticationToken;
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationToken;

import com.sap.cds.feature.auth.AuthenticatedUserClaimProvider;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.framework.spring.utils.SpringObjects;

public class SpringAuthenticatedUserClaimProvider implements AuthenticatedUserClaimProvider {

	private static boolean securityContextAvailable;
	private static boolean oauth2ClientAvailable;
	private static boolean oauth2ResourceAvailable;
	private static boolean basicAuthAvailable;

	static {
		try {
			securityContextAvailable = SecurityContextHolder.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			securityContextAvailable = false;
		}

		try {
			oauth2ClientAvailable = OAuth2AuthenticationToken.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			oauth2ClientAvailable = false;
		}

		try {
			oauth2ResourceAvailable = JwtAuthenticationToken.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			oauth2ResourceAvailable = false;
		}

		try {
			basicAuthAvailable = UsernamePasswordAuthenticationToken.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			basicAuthAvailable = false;
		}

	}

	@Override
	public String getUserClaim() {
		String token = null;

		if(securityContextAvailable) {
			Authentication auth = SecurityContextHolder.getContext().getAuthentication();

			if (basicAuthAvailable) {
				if (token == null && auth instanceof UsernamePasswordAuthenticationToken) { // NOSONAR
					Object principal = ((UsernamePasswordAuthenticationToken) auth).getPrincipal();
					if (principal instanceof org.springframework.security.core.userdetails.User) {
						token = ((org.springframework.security.core.userdetails.User) principal).getUsername();
					}
				}
			}

			if(oauth2ClientAvailable) {
				if(token == null && auth instanceof OAuth2AuthenticationToken) {
					// oauth2 client
					String authorizedClientRegistrationId = ((OAuth2AuthenticationToken) auth).getAuthorizedClientRegistrationId();
					String principalName = ((OAuth2AuthenticationToken) auth).getName();

					for(OAuth2AuthorizedClientService clientService : SpringObjects.getBeans(OAuth2AuthorizedClientService.class)) {
						OAuth2AuthorizedClient authorizedClient = clientService.loadAuthorizedClient(authorizedClientRegistrationId, principalName);
						if(authorizedClient != null) {
							token = authorizedClient.getAccessToken().getTokenValue();
							break;
						}
					}
				}
			}

			if(oauth2ResourceAvailable) {
				if (token == null && auth instanceof JwtAuthenticationToken) {
					// oauth2 resource server
					token = ((JwtAuthenticationToken) auth).getToken().getTokenValue();
				}
			}
		}

		return token;
	}

	@Override
	public boolean isActiveFeature() {
		try {
			boolean mockUsersAvailable = !Properties.getCds().getSecurity().getMock().getUsers().isEmpty();
			return securityContextAvailable && (oauth2ClientAvailable || oauth2ResourceAvailable || (basicAuthAvailable && mockUsersAvailable));
		} catch(Exception e) { // NOSONAR
			return false;
		}
	}

	@Override
	public String getFeatureName() {
		return "Spring Authentication Claim Provider";
	}
}
