/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.actuator;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;

import com.google.common.collect.Lists;
import com.sap.cds.adapter.ServletUrlResourcePaths;
import com.sap.cds.adapter.ServletUrlResourcePaths.UrlResourcePathVisitor;
import com.sap.cds.adapter.UrlResourcePath;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.config.pojo.CdsProperties.Security;
import com.sap.cds.framework.spring.config.auth.MockUsersSecurityConfig;
import com.sap.cds.framework.spring.config.auth.XsuaaSecurityConfig;
import com.sap.cds.framework.spring.mt.MtFeature;
import com.sap.cds.framework.spring.utils.SpringObjects;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.info.CdsInfo;

@Configuration
@ConditionalOnClass(WebSecurityConfigurer.class) // only if spring-security library has been loaded
public class CdsSecurityInfo {

	enum Authentication {
		NONE ("Not configured", null, "All endpoints are public!", null),
		CUSTOM ("Custom", "<unknown>", "Make sure CDS adapter endpoints are secured accordingly", null),
		MOCK ("Mock users", "Basic", "Not appropriate for productive usage!", MockUsersSecurityConfig.class),
		XSUAA ("XSUAA", "OAuth2", null, XsuaaSecurityConfig.class);

		private final String name;
		private final String type;
		private final String remark;
		private final Class<?> config;

		private Authentication(String name, String type, String remark, Class<?> config) {
			this.name = name;
			this.type = type;
			this.remark = remark;
			this.config = config;
		}
	}

	private Authentication authentication;

	private CdsRuntime cdsRuntime;

	private MtFeature mtFeature;

	@Autowired
	public CdsSecurityInfo(List<WebSecurityConfigurerAdapter> adapters, CdsRuntime cdsRuntime, MtFeature mtFeature) {

		this.cdsRuntime = cdsRuntime;
		this.mtFeature = mtFeature;
		this.authentication = Authentication.NONE;
		if (adapters != null) {
			// find an Authentication with matching class in the security adapters (there should be zero or one)
			this.authentication = Stream.of(Authentication.values()).filter(
					e -> e.config != null &&
					adapters.stream().filter(a -> e.config.isAssignableFrom(a.getClass())).findAny().isPresent()).findFirst()
					.orElse(Authentication.CUSTOM);
		}
	}


	@SuppressWarnings("serial")
	private Map<String, Object> authentication() {
		return new LinkedHashMap<String, Object>() {{
			put("name", authentication.name);
			if (authentication.type != null) {
				put("type", authentication.type);
			}
			if (authentication.remark != null) {
				put("remark", authentication.remark);
			}
		}};
	}

	@SuppressWarnings("serial")
	private Map<String, Object> authorization() {
		return new LinkedHashMap<String, Object>() {{
			Security security = Properties.getCds().getSecurity();
			put("authenticate-unknown-endpoints", security.isAuthenticateUnknownEndpoints());
			put("instance-based-authorization", security.getInstanceBasedAuthorization().isEnabled());
			put("open-unrestricted-endpoints", security.getOpenUnrestrictedEndpoints(mtFeature.isActive()));
			put("draft-protection", security.getDraftProtection().isEnabled());
		}};
	}

	private Map<String, List<String>> publicEndpoints() {

		Map<String, List<String>> result = new LinkedHashMap<>();

		if (authentication == Authentication.MOCK || authentication == Authentication.XSUAA) {
			if (Properties.getCds().getSecurity().getOpenUnrestrictedEndpoints(mtFeature.isActive())) {
				new ServletUrlResourcePaths(cdsRuntime).visit(new UrlResourcePathVisitor() {

					@Override
					public void foundPublicPath(UrlResourcePath publicPath) {
						result.put(publicPath.getPath(), Lists.newArrayList("*"));
					}

					@Override
					public void foundPublicEvents(UrlResourcePath path, Stream<String> publicEvents) {
						List<String> publicEventList = publicEvents.collect(Collectors.toList());
						if (!publicEventList.isEmpty()) {
							result.put(path.getPath(), publicEventList);
						}
					}

				});
			}
		}

		return result;
	}

	public static class CdsSecurityInfoImpl implements CdsInfo {

		@SuppressWarnings("serial")
		@Override
		public Map<String, Object> info(Details details) {
			CdsSecurityInfo securityInfo = SpringObjects.getBeans(CdsSecurityInfo.class).stream().findFirst().orElse(null);
			return new LinkedHashMap<String, Object>() {{
				if (securityInfo != null) {
					put("authentication", securityInfo.authentication());
					put("authorization", securityInfo.authorization());
					if (details == Details.HIGH) {
						Map<String, List<String>> publicEndpoints = securityInfo.publicEndpoints();
						if (!publicEndpoints.isEmpty()) {
							put("public CDS adapter endpoints", publicEndpoints);
						}
					}
				} else {
					// can't make use of Authentication.NONE which is not loadable
					put("authentication", new LinkedHashMap<String, Object>() {{
						put("name", "Not configured");
						put("remark", "All endpoints are public!");
					}});
				}
			}};
		}

		@Override
		public String name() {
			return "security";
		}
	}

}
