package com.sap.cds.framework.spring.config.auth;

import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpMethod;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.ExpressionUrlAuthorizationConfigurer;

import com.sap.cds.adapter.ServletUrlResourcePaths;
import com.sap.cds.adapter.ServletUrlResourcePaths.UrlResourcePathVisitor;
import com.sap.cds.adapter.UrlResourcePath;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.framework.spring.mt.MtFeature;
import com.sap.cds.services.runtime.CdsRuntime;

@Configuration
@ConditionalOnClass(WebSecurityConfigurer.class) // only if spring-security library has been loaded
public class CdsHttpSecurityConfigurerConfig {

	@Bean
	@ConditionalOnMissingBean  // Auto configuration classes are guaranteed to be loaded after any user-defined bean definitions, hence no ordering is required
	public HttpSecurityConfigurer httpSecurityConfigurer(CdsRuntime runtime, MtFeature mtFeature) {
		return new CdsModelBasedSecurityConfigurer(runtime, mtFeature);
	}

	/**
	 * An HttpSecurityConfigurer which is based on the service model of the underlying CDS model.
	 * Endpoints that are not authorized with @requires or @restrict are not authenticated by default.
	 */
	private static class CdsModelBasedSecurityConfigurer implements HttpSecurityConfigurer {

		private final static Logger logger = LoggerFactory.getLogger(HttpSecurityConfigurer.class);

		private final CdsRuntime runtime;
		private final MtFeature mtFeature;

		public CdsModelBasedSecurityConfigurer(CdsRuntime runtime, MtFeature mtFeature) {
			this.runtime = runtime;
			this.mtFeature = mtFeature;
		}

		@Override
		public void configure(HttpSecurity http) throws Exception { // NOSONAR

			ServletUrlResourcePaths servletPaths = new ServletUrlResourcePaths(runtime);

			if( !Properties.getCds().getSecurity().isAuthenticateUnknownEndpoints() ) {
				// narrow the base urls to the urls of the servlets
				http.requestMatchers().antMatchers(servletPaths.getBasePaths().map(UrlResourcePath::getPath).toArray(String[]::new));
				logger.info("Configuring authentication of CDS adapter endpoints. Other endpoints are not configured.");
			}

			ExpressionUrlAuthorizationConfigurer<HttpSecurity>.ExpressionInterceptUrlRegistry urlRegistry = http.authorizeRequests();

			if ( Properties.getCds().getSecurity().getOpenUnrestrictedEndpoints(mtFeature.isActive()) ) {
				// open endpoints which are declared as public
				servletPaths.visit(new UrlResourcePathVisitor() {

					@Override
					public void foundPublicPath(UrlResourcePath publicPath) {
						urlRegistry.antMatchers(publicPath.getPath()).permitAll();
						logger.debug("Public CDS endpoint {}", publicPath.getPath());
					}

					@Override
					public void foundPublicEvents(UrlResourcePath path, Stream<String> publicEvents) {
						publicEvents.forEach( publicEvent -> {
							urlRegistry.antMatchers(HttpMethod.valueOf(publicEvent), path.getPath()).permitAll();
							logger.debug("Public CDS endpoint {} {}", publicEvent, path.getPath());
						});

						urlRegistry.antMatchers(path.getPath()).authenticated(); // Required as parent path may be open with recursive pattern
						logger.debug("Authenticate CDS endpoint {}", path.getPath());
					}

				});
			} else {
				logger.info("Disabled configuration of public CDS adapter endpoints. All CDS adapter endpoints require authentication.");
			}

			// all other endpoints of the servlets are closed by default
			urlRegistry.anyRequest().authenticated();
		}
	}

}
