/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.authentication.configurers.provisioning.InMemoryUserDetailsManagerConfigurer;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.crypto.factory.PasswordEncoderFactories;
import org.springframework.security.crypto.password.PasswordEncoder;

import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.config.pojo.CdsProperties.Security.Mock.User;

/**
 * The default spring security configuration based on mock users. Active if and only if mock users are configured in the active profile.
 * Mock users are authenticated via basic authentication. By default, passwords are empty.
 */
@Configuration
@ConditionalOnClass(WebSecurityConfigurer.class) // only if spring-security library has been loaded
@AutoConfigureAfter(XsuaaSecurityConfig.class) // XsuaaSecurityConfig has higher priority and should be checked first
@ConditionalOnMissingBean(XsuaaSecurityConfig.class) // don't activate mock users in case of XSUAA config
@Conditional(MockUsersConfiguredCondition.class) // only load in case mock users are configured
@EnableWebSecurity
@Order(1000)
public class MockUsersSecurityConfig extends WebSecurityConfigurerAdapter {

	private final static Logger logger = LoggerFactory.getLogger(MockUsersSecurityConfig.class);

	@Autowired
	private HttpSecurityConfigurer httpSecurityConfigurer;

	@Override
	public void configure(AuthenticationManagerBuilder auth) throws Exception {

		List<User> users = Properties.getCds().getSecurity().getMock().getUsers();
		if (users.isEmpty()) {
			// go with standard spring security auth manager if there is no mock user configured (i.e. there will be default user 'user' with password written to console)
			logger.warn("No mock users found in configuration. Spring standard authentication manager with default user 'user' (password written to console) is available");
			super.configure(auth);
			return;
		}

		InMemoryUserDetailsManagerConfigurer<AuthenticationManagerBuilder> authConfigurer = auth.inMemoryAuthentication();
		PasswordEncoder encoder = PasswordEncoderFactories.createDelegatingPasswordEncoder();
		authConfigurer.passwordEncoder(encoder);
		for (User user : users) {
			if (!user.isValid()) {
				logger.warn("Skipping invalid mock user defintion {} ('name' and 'password' required)", user);
			} else {
				authConfigurer.withUser(user.getName()).password(encoder.encode(user.getPassword())).authorities(user.getRoles().toArray(new String[0]));
				logger.info("Added mock user {}", user);
			}
		}
	}

	@Override
	public void configure(HttpSecurity http) throws Exception {

		http.csrf().disable();

		httpSecurityConfigurer.configure(http);

		http.httpBasic();

		logger.info("*************************************************************************");
		logger.info("*  Security configuration based on mock users found in active profile.  *");
		logger.info("*                 !!! NEVER USE IN PRODUCTIVE MODE !!!                  *");
		logger.info("*************************************************************************");
	}

}
