/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.Ordered;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.web.servlet.handler.SimpleUrlHandlerMapping;

import com.sap.cds.adapter.AdapterFactory;
import com.sap.cds.adapter.ServletAdapterFactory;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.services.db.DataSourceDescriptor;
import com.sap.cds.feature.services.util.ServiceDescriptorUtils;
import com.sap.cds.framework.spring.config.adapter.AdapterBeanFactory;
import com.sap.cds.framework.spring.servicedescriptor.DataSourceConnector;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;

@Configuration
public class CdsRuntimeConfig implements ImportBeanDefinitionRegistrar, EnvironmentAware {

	private static final Logger logger = LoggerFactory.getLogger(CdsRuntimeConfig.class);
	private static boolean springWebAvailable;

	static {
		try {
			springWebAvailable = SimpleUrlHandlerMapping.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			springWebAvailable = false;
		}
	}

	private Environment environment;

	@Override
	public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
		CdsRuntimeConfigurer configurer = CdsRuntimeConfigurer.create()
				.cdsModel()
				.serviceConfigurations();

		BeanDefinitionBuilder configurerBuilder = BeanDefinitionBuilder.genericBeanDefinition(SimpleBeanFactory.class);
		configurerBuilder.addConstructorArgValue(configurer);
		registry.registerBeanDefinition("cdsRuntimeConfigurer", configurerBuilder.getBeanDefinition());

		CdsRuntime runtime = configurer.getCdsRuntime();
		BeanDefinitionBuilder runtimeBuilder = BeanDefinitionBuilder.genericBeanDefinition(SimpleBeanFactory.class);
		runtimeBuilder.addConstructorArgValue(runtime);
		registry.registerBeanDefinition("cdsRuntime", runtimeBuilder.getBeanDefinition());

		registerDataSources(registry, runtime);
		registerServices(registry, runtime);
		registerAdapters(registry, runtime);
	}

	private void registerDataSources(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		// only configure if no spring datasource is configured, otherwise the configuration of spring.datasource would be ignored
		String springDataSourceUrl = environment.getProperty("spring.datasource.url");
		if(springDataSourceUrl != null) {
			// check for an embedded SQLite database
			if(springDataSourceUrl.contains(":sqlite:") && springDataSourceUrl.contains(":memory:")) {
				Properties.getCds().getDataSource().setEmbedded(true);
			}
			logger.info("Found 'spring.datasource.url' configuration: Auto-configuration of DataSource beans is disabled.");
			return;
		}

		// get all datasource based service descriptors available in the environment
		List<DataSourceDescriptor> descriptors = ServiceDescriptorUtils.getDataSourceDescriptors();

		for(DataSourceDescriptor descriptor : descriptors) {
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(DataSourceConnector.class);
			boolean isPrimary = descriptor.getName().equals(Properties.getCds().getDataSource().getBinding());

			builder.addConstructorArgValue(descriptor);
			builder.addConstructorArgValue(runtime);
			builder.setPrimary(isPrimary);
			registry.registerBeanDefinition(descriptor.getName(), builder.getBeanDefinition());
			logger.info("Registered {}DataSource '{}'", isPrimary ? "primary " : "", descriptor.getName());

			if(isPrimary && descriptor.isEmbedded()) {
				Properties.getCds().getDataSource().setEmbedded(true);
			}
		}
	}

	private void registerServices(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		runtime.getServiceCatalog().getServices().forEach((service) -> {
			BeanDefinitionBuilder serviceBuilder = BeanDefinitionBuilder.genericBeanDefinition(SimpleBeanFactory.class);
			serviceBuilder.addConstructorArgValue(service);
			registry.registerBeanDefinition(service.getName(), serviceBuilder.getBeanDefinition());
		});
	}

	private void registerAdapters(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		Iterator<AdapterFactory> factoryIterator = ServiceLoader.load(AdapterFactory.class).iterator();
		while(factoryIterator.hasNext()) {
			AdapterFactory factory = factoryIterator.next();
			// skip initialization of servlet adapters, if spring-web is not available
			if(factory instanceof ServletAdapterFactory && !springWebAvailable) {
				continue;
			}

			if(factory.isEnabled()) {
				// create a bean for the adapter
				String beanName = factory.getClass().getSimpleName();
				BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(AdapterBeanFactory.class);
				builder.addConstructorArgValue(factory);
				registry.registerBeanDefinition(beanName, builder.getBeanDefinition());

				// create a URL mapping in case of a servlet adapter
				if(factory instanceof ServletAdapterFactory) {
					ServletAdapterFactory servletFactory = (ServletAdapterFactory) factory;

					BeanDefinitionBuilder mappingBuilder = BeanDefinitionBuilder.genericBeanDefinition(SimpleUrlHandlerMapping.class, () -> {
						SimpleUrlHandlerMapping mapping = new SimpleUrlHandlerMapping();
						Map<String, String> urlMap = new HashMap<>();
						// TODO ServiceCatalog might not be fully populated with custom created ApplicationService instances
						for(String m : servletFactory.getMappings(runtime)) {
							urlMap.put(m, beanName);
						}
						mapping.setOrder(Ordered.LOWEST_PRECEDENCE - 1);
						mapping.setUrlMap(urlMap);
						return mapping;
					});
					registry.registerBeanDefinition(beanName + "Mapping", mappingBuilder.getBeanDefinition());
				}
			}
		}
	}

	@Override
	public void setEnvironment(Environment environment) {
		this.environment = environment;
	}

}
