/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.servicedescriptor;

import static com.sap.cds.feature.services.util.DataSourceProviderUtils.PoolType.DBCP2;
import static com.sap.cds.feature.services.util.DataSourceProviderUtils.PoolType.HIKARI;
import static com.sap.cds.feature.services.util.DataSourceProviderUtils.PoolType.TOMCAT;

import java.util.Arrays;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.boot.jdbc.DataSourceBuilder;

import com.sap.cds.feature.services.db.DataSourceDescriptor;
import com.sap.cds.feature.services.db.DataSourceProvider;
import com.sap.cds.feature.services.util.DataSourceProviderUtils;
import com.sap.cds.framework.spring.utils.SpringObjects;

/**
 * The default provider for data sources.
 */
public class SpringDataSourceProvider implements DataSourceProvider {

	@Override
	public DataSource create(DataSourceDescriptor descriptor) {
		DataSourceBuilder<?> builder = DataSourceBuilder.create();
		builder.driverClassName(descriptor.getDriverClassName());
		builder.url(descriptor.getUrl());
		builder.username(descriptor.getUsername());
		builder.password(descriptor.getPassword());
		DataSource ds = builder.build();

		Binder binder = Binder.get(SpringObjects.getEnvironment());
		BindResult<DataSource> bindResult;
		Bindable<DataSource> bindableDs = Bindable.ofInstance(ds);
		for (String section: getDataSourceSections(descriptor)) {
			bindResult = binder.bind(section, bindableDs);
			if (bindResult.isBound()) {
				return bindResult.get();
			}
		}
		return ds;
	}

	private static List<String> getDataSourceSections(DataSourceDescriptor descriptor) {
		return Arrays.asList(DataSourceProviderUtils.getDataSourceSection(descriptor, HIKARI)
				, DataSourceProviderUtils.getDataSourceSection(descriptor, TOMCAT)
				, DataSourceProviderUtils.getDataSourceSection(descriptor, DBCP2));
	}

	@Override
	public boolean accept(DataSourceDescriptor descriptor) {
		return descriptor.getDriverClassName() != null;
	}

}
