/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.slf4j.helpers.MessageFormatter;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.SpringApplicationRunListener;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceResolvable;
import org.springframework.context.NoSuchMessageException;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.Environment;

import com.sap.cds.services.runtime.CdsRuntimeConfigurer;

public class SpringObjects implements SpringApplicationRunListener  {

	private static ApplicationContext context;

	private static volatile Environment environment;

	private static volatile MessageSource messageSource = new MessageSource() {

		@Override
		public String getMessage(String code, Object[] args, String defaultMessage, Locale locale) {
			if (code != null) {
				return MessageFormatter.arrayFormat(code, args).getMessage();
			}
			return defaultMessage;
		}

		@Override
		public String getMessage(String code, Object[] args, Locale locale) throws NoSuchMessageException {
			return MessageFormatter.arrayFormat(code, args).getMessage();
		}

		@Override
		public String getMessage(MessageSourceResolvable resolvable, Locale locale) throws NoSuchMessageException {
			return resolvable.getDefaultMessage();
		}

	};

	public SpringObjects(SpringApplication application, String[] args) {
	}

	@Override
	public void environmentPrepared(ConfigurableEnvironment environment) {
		SpringObjects.environment = environment;
	}

	@Override
	public void started(ConfigurableApplicationContext context) {
		if(context.getBeanNamesForType(CdsRuntimeConfigurer.class).length == 0) return;

		SpringObjects.context = context;
		// directly load beans that are always available
		SpringObjects.messageSource = context.getBean(MessageSource.class);
	}

	public static Environment getEnvironment() {
		if(environment == null) {
			throw new IllegalStateException("Accessed environment before it was available");
		}
		return environment;
	}

	public static MessageSource getMessageSource() {
		return messageSource;
	}

	/**
	 * Loads beans which are optional and of which the class is not guaranteed to be on the class path
	 * @param <T> the type of the beans
	 * @param clazz the class of the beans
	 * @return the beans as list
	 */
	@SuppressWarnings("unchecked")
	public static <T> List<T> getBeans(Class<T> clazz) {
		if(context == null) {
			throw new IllegalStateException("Accessed context before it was available");
		}
		List<T> beans = new ArrayList<>();
		String[] beanNames = context.getBeanNamesForType(clazz);
		for(String beanName : beanNames) {
			beans.add((T) context.getBean(beanName));
		}
		return beans;
	}


}
