/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.function.Supplier;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.jdbc.DataSourceTransactionManagerAutoConfiguration;
import org.springframework.boot.jdbc.EmbeddedDatabaseConnection;
import org.springframework.boot.sql.init.dependency.DependsOnDatabaseInitialization;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.TransactionAwareDataSourceProxy;
import org.springframework.transaction.PlatformTransactionManager;

import com.sap.cds.CdsDataStore;
import com.sap.cds.framework.spring.transaction.PlatformTransactionManagerPostProcessor.SpringTransactionManagerGetter;
import com.sap.cds.framework.spring.utils.ProxyUtils;
import com.sap.cds.services.impl.persistence.JdbcPersistenceService;
import com.sap.cds.services.impl.persistence.JdbcPersistenceServiceConfiguration;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.transaction.TransactionManager;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

@Configuration
// only if cds-feature-jdbc, spring-jdbc and spring-tx are available
@ConditionalOnClass({ JdbcPersistenceServiceConfiguration.class, TransactionAwareDataSourceProxy.class, PlatformTransactionManager.class })
@ConditionalOnBean({ DataSource.class, PlatformTransactionManager.class }) // datasource and tx manager must exist
@AutoConfigureAfter({ DataSourceAutoConfiguration.class, DataSourceTransactionManagerAutoConfiguration.class})
@DependsOnDatabaseInitialization
public class SpringJdbcPersistenceServiceConfiguration {

	private static final Logger logger = LoggerFactory.getLogger(SpringJdbcPersistenceServiceConfiguration.class);
	private final JdbcPersistenceServiceConfiguration configuration;

	@Autowired
	public SpringJdbcPersistenceServiceConfiguration(DataSource dataSource, PlatformTransactionManager platformTransactionManager, CdsRuntime runtime) {
		// determine if datasource is considered embedded by Spring, if CSV initialization mode is set to embedded
		// the check attempts to create a datasource connection
		String csvInitMode = runtime.getEnvironment().getCdsProperties().getDataSource().getCsvInitializationMode();
		if("embedded".equals(csvInitMode) && isEmbedded(dataSource)) {
			logger.debug("Determined DataSource as embedded based on connection check.");
			runtime.getEnvironment().getCdsProperties().getDataSource().setEmbedded(true);
		}

		TransactionManager txMgr = ((SpringTransactionManagerGetter) platformTransactionManager).getSpringTransactionManager();
		DataSource txAwareDataSource = new TransactionAwareDataSourceProxy(dataSource);
		Supplier<Connection> connectionSupplier = () -> {
			try {
				return txAwareDataSource.getConnection();
			} catch (SQLException e) {
				throw new ErrorStatusException(CdsErrorStatuses.JDBC_CONNECTION_FAILED, e);
			}
		};

		this.configuration = JdbcPersistenceServiceConfiguration.create(connectionSupplier, txMgr);
	}

	@Bean(PersistenceService.DEFAULT_NAME)
	public JdbcPersistenceService persistenceService(CdsRuntimeConfigurer configurer) {
		return configuration.createOrGetService(configurer);
	}

	@Bean
	public CdsDataStore cdsDataStore(CdsRuntimeConfigurer configurer) {
		JdbcPersistenceService persistenceService = configuration.createOrGetService(configurer);
		return ProxyUtils.createProxy(CdsDataStore.class, () -> persistenceService.getCdsDataStore());
	}

	private boolean isEmbedded(DataSource dataSource) {
		try {
			return EmbeddedDatabaseConnection.isEmbedded(dataSource);
		} catch (Exception e) {
			return false;
		}
	}

}
