/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.Ordered;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.web.servlet.handler.SimpleUrlHandlerMapping;

import com.sap.cds.adapter.AdapterFactory;
import com.sap.cds.adapter.ServletAdapterFactory;
import com.sap.cds.framework.spring.config.adapter.AdapterBeanFactory;
import com.sap.cds.framework.spring.config.datasource.DataSourceBeanFactory;
import com.sap.cds.services.Service;
import com.sap.cds.services.datasource.DataSourceDescriptor;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.runtime.ExtendedServiceLoader;
import com.sap.cds.services.utils.datasource.DataSourceUtils;

@Configuration
public class CdsRuntimeConfig implements ImportBeanDefinitionRegistrar, EnvironmentAware {

	private static final Logger logger = LoggerFactory.getLogger(CdsRuntimeConfig.class);
	private static boolean springWebAvailable;

	static {
		try {
			springWebAvailable = SimpleUrlHandlerMapping.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			springWebAvailable = false;
		}
	}

	private Environment environment;

	@Override
	public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
		CdsRuntimeConfigurer configurer = BootstrapCache.get(environment)
				.serviceConfigurations();

		BeanDefinitionBuilder configurerBuilder = BeanDefinitionBuilder.genericBeanDefinition(CdsRuntimeConfigurer.class, () -> configurer);
		registry.registerBeanDefinition("cdsRuntimeConfigurer", configurerBuilder.getBeanDefinition());

		CdsRuntime runtime = configurer.getCdsRuntime();
		BeanDefinitionBuilder runtimeBuilder = BeanDefinitionBuilder.genericBeanDefinition(CdsRuntime.class, () -> runtime);
		registry.registerBeanDefinition("cdsRuntime", runtimeBuilder.getBeanDefinition());

		registerDataSources(registry, runtime);
		registerServices(registry, runtime);
		registerAdapters(registry, runtime);
	}

	@SuppressWarnings("unchecked")
	private void registerDataSources(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		// only configure if no spring datasource is configured, otherwise the configuration of spring.datasource would be ignored
		String springDataSourceUrl = environment.getProperty("spring.datasource.url");
		if(springDataSourceUrl != null) {
			// check for an embedded SQLite database
			if(springDataSourceUrl.contains(":sqlite:") && springDataSourceUrl.contains(":memory:")) {
				logger.debug("Determined DataSource as embedded based on 'spring.datasource.url' configuration.");
				runtime.getEnvironment().getCdsProperties().getDataSource().setEmbedded(true);
			}
			logger.info("Found 'spring.datasource.url' configuration: Auto-configuration of DataSource beans is disabled.");
			return;
		}

		if(!runtime.getEnvironment().getCdsProperties().getDataSource().getAutoConfig().isEnabled()) {
			logger.info("Auto-configuration of DataSource beans is explicitly disabled.");
			return;
		}

		// create datasources based on descriptors available in the environment
		List<DataSourceDescriptor> descriptors = DataSourceUtils.getDataSourceDescriptors(runtime);
		for(DataSourceDescriptor descriptor : descriptors) {
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(DataSourceBeanFactory.class);
			boolean isPrimary = descriptor.getName().equals(runtime.getEnvironment().getCdsProperties().getDataSource().getBinding());

			builder.addConstructorArgValue(descriptor);
			builder.addConstructorArgValue(environment);
			builder.setPrimary(isPrimary);
			registry.registerBeanDefinition(descriptor.getName(), builder.getBeanDefinition());
			logger.info("Registered {}DataSource '{}'", isPrimary ? "primary " : "", descriptor.getName());
		}

		// create datasources provided by datasource providers
		Map<String, DataSource> dataSources = DataSourceUtils.getDataSources(runtime);
		for(Map.Entry<String, DataSource> entry : dataSources.entrySet()) {
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition((Class<DataSource>) entry.getValue().getClass(), () -> entry.getValue());
			boolean isPrimary = entry.getKey().equals(runtime.getEnvironment().getCdsProperties().getDataSource().getBinding());
			builder.setPrimary(isPrimary);
			registry.registerBeanDefinition(entry.getKey(), builder.getBeanDefinition());
			logger.info("Registered {}DataSource '{}'", isPrimary ? "primary " : "", entry.getKey());
		}
	}

	@SuppressWarnings("unchecked")
	private void registerServices(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		runtime.getServiceCatalog().getServices().forEach((service) -> {
			BeanDefinitionBuilder serviceBuilder = BeanDefinitionBuilder.genericBeanDefinition((Class<Service>) service.getClass(), () -> service);
			registry.registerBeanDefinition(service.getName(), serviceBuilder.getBeanDefinition());
		});
	}

	private void registerAdapters(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		Iterator<AdapterFactory> factoryIterator = ExtendedServiceLoader.loadAll(AdapterFactory.class, runtime);
		while(factoryIterator.hasNext()) {
			AdapterFactory factory = factoryIterator.next();
			// skip initialization of servlet adapters, if spring-web is not available
			if(factory instanceof ServletAdapterFactory && !springWebAvailable) {
				continue;
			}

			if(factory.isEnabled()) {
				// create a bean for the adapter
				String beanName = factory.getClass().getSimpleName();
				BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(AdapterBeanFactory.class);
				builder.addConstructorArgValue(factory);
				registry.registerBeanDefinition(beanName, builder.getBeanDefinition());

				// create a URL mapping in case of a servlet adapter
				if(factory instanceof ServletAdapterFactory) {
					ServletAdapterFactory servletFactory = (ServletAdapterFactory) factory;

					BeanDefinitionBuilder mappingBuilder = BeanDefinitionBuilder.genericBeanDefinition(SimpleUrlHandlerMapping.class, () -> {
						SimpleUrlHandlerMapping mapping = new SimpleUrlHandlerMapping();
						Map<String, String> urlMap = new HashMap<>();
						// TODO ServiceCatalog might not be fully populated with custom created ApplicationService instances
						for(String m : servletFactory.getMappings()) {
							urlMap.put(m, beanName);
						}
						mapping.setOrder(Ordered.LOWEST_PRECEDENCE - 1);
						mapping.setUrlMap(urlMap);
						return mapping;
					});
					registry.registerBeanDefinition(beanName + "Mapping", mappingBuilder.getBeanDefinition());
				}
			}
		}
	}

	@Override
	public void setEnvironment(Environment environment) {
		this.environment = environment;
	}

}
