/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.mt;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnSingleCandidate;
import org.springframework.stereotype.Component;

import com.sap.cds.framework.spring.config.runtime.CdsRuntimeConfig;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.MtAsyncUnsubscribeEventContext;
import com.sap.cds.services.mt.MtSubscriptionService;
import com.sap.cds.services.mt.MtUnsubscribeEventContext;
import com.sap.cloud.mt.runtime.TenantAwareDataSource;

/**
 * AFTER handler for mt unsubscription events.
 */

@Component
@ConditionalOnClass(TenantAwareDataSource.class)
@ConditionalOnSingleCandidate(TenantAwareDataSource.class)
@AutoConfigureAfter(CdsRuntimeConfig.class)
@ServiceName(MtSubscriptionService.DEFAULT_NAME)
public class MtUnsubscriptionClearDatasourceHandler implements EventHandler {

	private static final Logger log = LoggerFactory.getLogger(MtUnsubscriptionClearDatasourceHandler.class);

	@Autowired
	private TenantAwareDataSource dataSource;

	@After
	protected void afterUnsubscribe(MtUnsubscribeEventContext context) {
		if (context.getDelete() != null && context.getDelete()) {
			removeDataSourceFromCache(context, context.getTenantId());
		}
	}

	@After
	protected void afterUnsubscribeAsync(MtAsyncUnsubscribeEventContext context) {
		if (context.getDelete() != null && context.getDelete()) {
			removeDataSourceFromCache(context, context.getTenantId());
		}
	}

	private void removeDataSourceFromCache(EventContext context, String tenantId) {
		if (dataSource != null) {
			log.debug("Removing data source of tenant '{}' from cache", tenantId);
			dataSource.deleteFromCache(tenantId);
		}
	}

}
