/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.xsuaa;

import static com.sap.cds.services.utils.XsuaaUtils.AuthenticationType.IAS_XSUAA;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.annotation.web.configurers.oauth2.server.resource.OAuth2ResourceServerConfigurer;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.web.client.RestOperations;

import com.sap.cds.framework.spring.config.auth.HttpSecurityConfigurer;
import com.sap.cds.framework.spring.config.auth.identity.IdentitySecurityConfig;
import com.sap.cds.services.environment.CdsProperties.Security.Xsuaa.Cache;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cds.services.utils.XsuaaUtils.AuthenticationType;
import com.sap.cloud.security.xsuaa.XsuaaServiceConfiguration;
import com.sap.cloud.security.xsuaa.client.OAuth2TokenService;
import com.sap.cloud.security.xsuaa.client.XsuaaOAuth2TokenService;
import com.sap.cloud.security.xsuaa.extractor.IasXsuaaExchangeBroker;
import com.sap.cloud.security.xsuaa.token.TokenAuthenticationConverter;
import com.sap.cloud.security.xsuaa.tokenflows.TokenCacheConfiguration;


/**
 * The default security configuration in case of active XSUAA configuration - which should be the case in productive scenarios.
 * By default, all requests require an oauth2 authentication - including public endpoints.
 * In order to open public endpoints, the application needs to create a security configuration with higher priority
 * and override these endpoints.
 */
@Configuration
@ConditionalOnClass({ XsuaaServiceConfiguration.class, WebSecurityConfigurer.class, Jwt.class }) // only if spring-xsuaa and spring-security web & JWT library has been loaded
@Conditional(XsuaaConfiguredCondition.class)
@AutoConfigureAfter(IdentitySecurityConfig.class)
@ConditionalOnMissingBean(IdentitySecurityConfig.class)
@EnableWebSecurity
@Order(1000)
public class XsuaaSecurityConfig extends WebSecurityConfigurerAdapter {

	private final static Logger logger = LoggerFactory.getLogger(XsuaaSecurityConfig.class);

	@Autowired
	private CdsRuntime runtime;
	
	@Autowired
	private XsuaaServiceConfiguration xsuaaServiceConfiguration;

	@Autowired
	private HttpSecurityConfigurer httpSecurityConfigurer;

	@Autowired
	private XsuaaFeature xsuaaFeatue;
	
	@Autowired
	private RestOperations xsuaaRestOperations;

	@Override
	protected void configure(HttpSecurity http) throws Exception {

		if ( !xsuaaFeatue.isActive() ) {
			throw new IllegalStateException("XSUAA configuration requires dependency to cds-feature-xsuaa. "
					+ "Alternatively, deactivate this security configuration with property 'cds.security.xsuaa.enabled: false' AND provide your own configuration.");
		}

		AuthenticationType authType = new XsuaaUtils(runtime).configuredAuthenticationType();
		
		// @formatter:off
		http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS); // state is kept in AppRouter

		httpSecurityConfigurer.configure(http);

		OAuth2ResourceServerConfigurer<HttpSecurity> resourceServer = http.oauth2ResourceServer();
		if (authType == IAS_XSUAA) {
			TokenCacheConfiguration tokenCacheConfiguration = TokenCacheConfiguration.cacheDisabled();
			Cache cache = runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().getCache();
			if (cache.isEnabled()) {
				tokenCacheConfiguration = TokenCacheConfiguration.getInstance(cache.getDuration(), cache.getSize(), cache.getTokenRemainingTime(),
						false); // cache statistics - we might later enable for cds-actuator
			}
			OAuth2TokenService oAuth2TokenService = new XsuaaOAuth2TokenService(xsuaaRestOperations, tokenCacheConfiguration);
			resourceServer.bearerTokenResolver(new IasXsuaaExchangeBroker(xsuaaServiceConfiguration, oAuth2TokenService)); // hybrid mode requires token exchange
		}
		resourceServer.jwt().jwtAuthenticationConverter(getJwtAuthoritiesConverter());
		// @formatter:on

		switch(authType) {
		case XSUAA:
			logger.info("Activated XSUAA security configuration");
			break;
		case IAS_XSUAA:
			logger.info("Activated XSUAA security configuration with IAS support");
			break;
		default:
			throw new IllegalStateException("Configuration error: Can't configure authentication without a binding to XSUAA service"); // this should not happen
		}
	}

	private Converter<Jwt, AbstractAuthenticationToken> getJwtAuthoritiesConverter() {
		TokenAuthenticationConverter converter = new TokenAuthenticationConverter(xsuaaServiceConfiguration);
		converter.setLocalScopeAsAuthorities(true);
		return converter;
	}

}
